source ./grammar.nu
source ./disc.nu

# session manager: [zellij](https://zellij.dev)
def "sessions" [] { zellij list-sessions | columnar | reject column1 | rename name age bind }
def recess [] { session end; sessions; }
def "session end" [] { zellij da -y }

def "session clean" [] {
  let session = session choose
  try { zellij kill-session $session }
  zellij delete-session $session
}

def "session choices" [] {
  ((ls ~/.config/zellij/session/ | get name) | path parse).stem
}

def "session choose" [] {
  ((session choices) ++ (sessions | get name)) | uniq | choose }

def "session running" []: nothing -> bool {
  $env | items {|k,s| $k} | any {|k| $k == ZELLIJ } }

def "session name" [name?:string]: nothing -> string {
  if ($name != null) { $name } else { basename $env.PWD | tr . _ } | str trim }

def "session has" [name?:string]: nothing -> bool {
  (session name $name) in (sessions).name }

def "session hop" [name?:string] {
  let name = $name | assure { session choose }
  zellij a -c $name options --default-layout (
    $"~/.config/zellij/session/($name).kdl" | path expand)
}

def "session record" [name?:string] {
  '~/space/rec' | mkcd
  let name = if ($name == null) { session choose } else { $name }
  asciinema rec --command $"nu -e 'session hop ($name)'" $"session.(day).(cloc).($name).rec"
}

def "session play" [] { '~/space/rec' | mkcd
  asciinema play ((ls).name | sort | choose) }

def hop [name?:string] { session hop $name }

def "session place" [p?: path, n?: string] {
  let place = if ($p | is-empty) { pwd } else { $p | path expand }
  let name = if ($n | is-empty) { $place | path basename } else { $n }
  let session = $"~/.config/zellij/session/($name).kdl" | path expand

  if ($session | path exists) {
    print $"Oh no; session name `($name)` is already in use.";
    if (input "Replace? (y/n)\t" | str downcase) != 'y' { return; }
  }

  echo $"layout { cwd \"($place)\"; pane; }" | save -f $session
}

def "session drop" [name: string] {
  let session = $"~/.config/zellij/session/($name).kdl" | path expand
  if ($session | path type) == file {
    open $session | print
    try { rm $session }
} }
