{ edge, lib, pkgs, ... } :
let
  nu = "${edge.nushell}/bin/nu";
  render-params = params: lib.concatMapStringsSep " " (param: ''"${param}"'') params;

basic = {
  shape = {
    nix = ''pane_template name="nix" { command "nix"; start_suspended true; };'';
  };

  # change = ''pane command="hx" borderless=true focus=true'';
  change = ''pane command="hx" borderless=true focus=true { args "."; }'';
  under = ''pane size=1 borderless=true { plugin location="zellij:status-bar"; }'';
  upper = ''pane size=1 borderless=true { plugin location="zellij:tab-bar"; }'';
  sudo = command: ''pane command="sudo" { args ${render-params command}; }'';
  nu = command: ''pane command="${nu}" { args "-e" "${command}"; }'';
  nup = command: ''pane command="${nu}" { args "-e" "${command}"; start_suspended true; }'';
  ready = command: ''pane command="${nu}" { args "-e" "${command}"; start_suspended true; }'';

  ns = pkgs: command: ''pane command="nix-shell" { args "-p" "${
  lib.concatStringsSep ''" "'' pkgs}" "${
  lib.replaceStrings [" "] [''" "''] command}";
  start_suspended true; }'';

  nd = command: ''pane command="nix" { args "develop" "--command" "${
  lib.replaceStrings [" "] [''" "''] command}";
  start_suspended true; };'';

  banners = inner: ''pane split_direction="horizontal" {
  ${inner}
  };'';

  columns = inner: ''pane split_direction="vertical" {
  ${inner}
  };'';
};

plans = {
  # oh no! lib is less-used, so should slide behind `basic` in call sequence.
  bird = (import ../session/bird.nix lib basic);

  # language-bound
  gleam = (import ../session/gleam.nix basic);
  sip = (import ../session/sip.nix basic);
  snake = (import ../session/snake.nix basic);
  slice = (import ../session/slice.nix basic);

  # machine
  exercise = (import ../session/exercise.nix basic);
  clean = (import ../session/clean.nix basic);

  # gram
  places = map: ({
    home.file =
      lib.attrsets.concatMapAttrs (name: node: ({
        ".config/zellij/session/${name}.kdl".text = ''
        layout { cwd "${node}";
          ${basic.upper}
          pane;
          ${basic.under}
        } '';
      })) map;
    });
};

build_session = { name, plan, lines ? [], base ? name }: {
  home.file.".config/zellij/session/${name}.kdl".text =
    plan { inherit base lines; };
};

in {
  home.packages = [ edge.zellij ];

  # enough is enough. come back and clean up more soon.
  imports = let
    session = name: body:
      { home.file.".config/zellij/session/${name}.kdl".text = body; };
  in [
    (plans.places {
      "board" = "~/disc/nvme0n1p6/gram/board";
      "gear" = "~/gear";
      # "group" = "~/group";
      # "issue" = "~/issue";
      "ledge" = "~/ledge";
      # "page" = "~/disc/nvme0n1p6/gram/page";
      "page" = "~/page";
      "gram" = "~/disc/nvme0n1p6/gram";
      "ramp" = "~/disc/nvme0n1p6/gram/ramp";
      "qu" = "~/disc/nvme0n1p6/gram/qu";

      # "dash" = "~/disc/nvme0n1p6/gram/dash";
      # "gloss" = "~/disc/nvme0n1p6/gram/gloss";
      # "min" = "~/disc/nvme0n1p6/gram/min";
      # "nuer" = "~/disc/nvme0n1p6/gram/nuer";
      # "page" = "~/disc/nvme0n1p6/gram/page";
      # "scope" = "~/disc/nvme0n1p6/gram/scope";

      "build" = "~/.build";
      "resume" = "~/resume";

      # mirrors
      # "nu" = "~/code/nushell/nushell";
      # "sync" = "~/code/rebase.hub/josevalim/sync";
      # "pen" = "~/base/nixpkgs/pkgs/by-name/pe/penpot";
    })

    (build_session { name = "blade"; plan = plans.slice; })
    # (build_session { name = "medi"; plan = plans.snake; })
    (build_session { name = "panel"; plan = plans.gleam; })
    # (build_session { name = "lens"; plan = plans.gleam; })
    # (build_session { name = "press"; plan = plans.bird; })

    # (build_session { name = "clean-code"; base = "~/base/code/"; plan = plans.clean; })
    # (build_session { name = "clean-code"; base = "~/disc/nvme0n1p6/gram/"; plan = plans.clean; })
    # (build_session { name = "clean-build"; base = "~/.build/"; plan = plans.clean; })
    # (build_session { name = "clean-home"; base = "~/"; plan = plans.clean; })

    (session "home" "layout { pane; }")
    (session "aurora" "layout { ${basic.nu "d -b nvme0n1p6; mkcd gram/aurora"}; }")
  ];

  home.file = {
    ".config/zellij/session/share.kdl".text = ''
        layout { cwd "~/disc/nvme0n1p6/gram/share";
          ${basic.banners ''
            ${basic.change}
            pane
            pane command="git" { args "push" "deploy" "main"; start_suspended true; }
          ''}
        }
      '';

    ".config/zellij/session/hub.kdl".text = ''
      layout { ${basic.columns ''
        pane command="nu" { args "-e" "d -b nvme0n1p6; cd code/tinyplex/tinyhub/client; nd bun run dev"; }
        pane start_suspended=true command="nu" { args "-e" "brave http://localhost:5173"; }
      ''} }'';

    ".config/zellij/session/read.kdl".text = ''
      layout { ${basic.columns ''
        pane command="nu" { args "-e" "nsh circumflex clx"; }
        pane command="nu" { args "-e" "nsh nom"; }
      ''} }'';

    ".config/zellij/session/mesh.kdl".text = ''
      layout { ${basic.columns ''
        pane command="nu" { args "-e" "mesh $m.baseboard"; }
      ''} }'';

    ".config/zellij/session/media.kdl".text = ''
      layout {
        pane command="nu" { args "-e" "disc load card/media"; }
        pane start_suspended=true command="nu" { args "-e" "mpv ...(glob 'card/media/youtube.com/@localfirstfm/#*.mkv' | sort -n | slice 13..-1)"; }
        pane start_suspended=true command="nu" { args "-e" "mpv ...(glob 'card/media/youtube.com/@localfirstconf/*.mkv')"; }
      }'';

    ".config/zellij/session/signal.kdl".text = ''
        layout { cwd "~";
          ${basic.ns [ "wireshark" ] "wireshark"}
      }'';

      # session: build
      # pane command="sudo" { args ${nu} "-e" "nixos-rebuild switch"; }
      # ${basic.nup "colmena apply --on base --build-on-target switch --impure"}

    ".config/zellij/session/log.kdl".text = ''
        layout { // cwd "~/disk";
          ${basic.columns ''
            ${basic.nu "log proxy op"};
            ${basic.nu "log proxy share"};
          ''}
        }
      '';

    ".config/zellij/session/image.kdl".text = ''
        layout { cwd "~/disk";
          ${basic.banners ''
            ${basic.nu "disc ls"};
            ${basic.nu "ls"};
          ''}
        }
      '';

    ".config/zellij/session/issue.kdl".text = ''
        layout { cwd "~/issue"; ${basic.change}; }
      '';

    ".config/zellij/session/muse.kdl".text = ''
        layout { ${basic.nu "muse"}; }
      '';

    ".config/zellij/session/record.kdl".text = ''
        layout { cwd "~/disc/nvme0n1p6/rec";
          ${basic.ready "nsh audio-recorder"};
          ${basic.columns ''
            ${basic.ready "screen record"};
            ${basic.ready "screen record"};
          ''}
        }
      '';
          # ${basic.nu "cam align 0 ...(cam shape blob)"};
          # ${basic.nu "sleep 4sec; cam screen 0"};
  };
}

# More zellij `config.kdl` options:

# // Choose what to do when zellij receives SIGTERM, SIGINT, SIGQUIT or SIGHUP
# // eg. when terminal window with an active zellij session is closed
# // Options:
# //   - detach (Default)
# //   - quit
# //
# // on_force_close "quit"

# // Send a request for a simplified ui (without arrow fonts) to plugins
# // simplified_ui true

# // Choose the path to the default shell that zellij will use for opening new panes
# // Default: $SHELL
# // default_shell ${nu}

# // Choose the path to override cwd that zellij will use for opening new panes
# //
# // default_cwd ""

# // Toggle between having Zellij lay out panes according to a predefined set of layouts whenever possible
# // auto_layout true

# // Whether sessions should be serialized to the cache folder (including their tabs/panes, cwds and running commands) so that they can later be resurrected
# // session_serialization false

# // Whether pane viewports are serialized along with the session, default is false
# // serialize_pane_viewport true

# // Scrollback lines to serialize along with the pane viewport when serializing sessions, 0
# // defaults to the scrollback size. If this number is higher than the scrollback size, it will
# // also default to the scrollback size. This does nothing if `serialize_pane_viewport` is not true.
# //
# // scrollback_lines_to_serialize 10000

# // Define color themes for Zellij
# // For more examples, see: https://github.com/zellij-org/zellij/tree/main/example/themes
# // Once these themes are defined, one of them should to be selected in the "theme" section of this file
# //
# // themes {
# //     dracula {
# //         fg 248 248 242
# //         bg 40 42 54
# //         red 255 85 85
# //         green 80 250 123
# //         yellow 241 250 140
# //         blue 98 114 164
# //         magenta 255 121 198
# //         orange 255 184 108
# //         cyan 139 233 253
# //         black 0 0 0
# //         white 255 255 255
# //     }
# // }

# // Choose the theme that is specified in the themes section.
# // Default: default
# //
# // theme "default"

# // The name of the default layout to load on startup
# // Default: "default"
# // default_layout "compact"

# // Choose the mode that zellij uses when starting up.
# // Default: normal
# //
# // default_mode "locked"

# // Toggle enabling the mouse mode.
# // On certain configurations, or terminals this could
# // potentially interfere with copying text.
# // Options:
# //   - true (default)
# //   - false
# //
# // mouse_mode false

# // Configure the scroll back buffer size
# // This is the number of lines zellij stores for each pane in the scroll back
# // buffer. Excess number of lines are discarded in a FIFO fashion.
# // Valid values: positive integers
# // Default value: 10000
# //
# // scroll_buffer_size 10000

# // Provide a command to execute when copying text. The text will be piped to
# // the stdin of the program to perform the copy. This can be used with
# // terminal emulators which do not support the OSC 52 ANSI control sequence
# // that will be used by default if this option is not set.
# // Examples:
# //
# // copy_command "xclip -selection clipboard" // x11
# // copy_command "wl-copy"                    // wayland
# // copy_command "pbcopy"                     // osx

# // Choose the destination for copied text
# // Allows using the primary selection buffer (on x11/wayland) instead of the system clipboard.
# // Does not apply when using copy_command.
# // Options:
# //   - system (default)
# //   - primary
# //
# // copy_clipboard "primary"

# // Enable or disable automatic copy (and clear) of selection when releasing mouse
# // Default: true
# //
# // copy_on_select false

# // When attaching to an existing session with other users,
# // should the session be mirrored (true)
# // or should each user have their own cursor (false)
# // Default: false
# //
# // mirror_session true

# // The folder in which Zellij will look for themes
# //
# // theme_dir "/path/to/my/theme_dir"

# // Enable or disable the rendering of styled and colored underlines (undercurl).
# // May need to be disabled for certain unsupported terminals
# // Default: true
# //
# // styled_underlines false
