// @observablehq/inputs v0.12.0 Copyright 2021–2024 Observable, Inc.
import { html } from 'htl';
import { format, parse } from 'isoformat';

function length(x) {
  return x == null ? null : typeof x === "number" ? `${x}px` : `${x}`;
}

function maybeWidth(width) {
  return {"--input-width": length(width)};
}

const bubbles = {bubbles: true};

function preventDefault(event) {
  event.preventDefault();
}

function dispatchInput({currentTarget}) {
  (currentTarget.form || currentTarget).dispatchEvent(new Event("input", bubbles));
}

function checkValidity(input) {
  return input.checkValidity();
}

function identity(x) {
  return x;
}

let nextId = 0;

function newId() {
  return `inputs-3a86ea-${++nextId}`;
}

function maybeLabel(label, input) {
  if (!label) return;
  label = html`<label>${label}`;
  if (input !== undefined) label.htmlFor = input.id = newId();
  return label;
}

function button(content = "≡", {
  label = "",
  value,
  reduce,
  disabled,
  required = false,
  width
} = {}) {
  const solitary = typeof content === "string" || content instanceof Node;
  if (solitary) {
    if (!required && value === undefined) value = 0;
    if (reduce === undefined) reduce = (value = 0) => value + 1;
    disabled = new Set(disabled ? [content] : []);
    content = [[content, reduce]];
  } else {
    if (!required && value === undefined) value = null;
    disabled = new Set(disabled === true ? Array.from(content, ([content]) => content) : disabled || undefined);
  }
  const form = html`<form class=inputs-3a86ea>`;
  form.addEventListener("submit", preventDefault);
  const style = {width: length(width)};
  const buttons = Array.from(content, ([content, reduce = identity]) => {
    if (typeof reduce !== "function") throw new TypeError("reduce is not a function");
    return html`<button disabled=${disabled.has(content)} style=${style} onclick=${event => {
      form.value = reduce(form.value);
      dispatchInput(event);
    }}>${content}`;
  });
  if (label = maybeLabel(label, solitary ? buttons[0] : undefined)) form.append(label);
  form.append(...buttons);
  form.value = value;
  return form;
}

function arrayify(array) {
  return Array.isArray(array) ? array : Array.from(array);
}

function iterable(array) {
  return array ? typeof array[Symbol.iterator] === "function" : false;
}

function maybeColumns(data) {
  if (iterable(data.columns)) return data.columns; // d3-dsv, FileAttachment
  if (data.schema && iterable(data.schema.fields)) return Array.from(data.schema.fields, f => f.name); // apache-arrow
  if (typeof data.columnNames === "function") return data.columnNames(); // arquero
}

// Note: use formatAuto (or any other localized format) to present values to the
// user; stringify is only intended for machine values.
function stringify(x) {
  return x == null ? "" : `${x}`;
}

const formatLocaleAuto = localize(locale => {
  const formatNumber = formatLocaleNumber(locale);
  return value => value == null ? ""
    : typeof value === "number" ? formatNumber(value)
    : value instanceof Date ? formatDate$1(value)
    : `${value}`;
});

const formatLocaleNumber = localize(locale => {
  return value => value === 0 ? "0" : value.toLocaleString(locale); // handle negative zero
});

const formatAuto = formatLocaleAuto();

const formatNumber = formatLocaleNumber();

function formatTrim(value) {
  const s = value.toString();
  const n = s.length;
  let i0 = -1, i1;
  out: for (let i = 1; i < n; ++i) {
    switch (s[i]) {
      case ".": i0 = i1 = i; break;
      case "0": if (i0 === 0) i0 = i; i1 = i; break;
      default: if (!+s[i]) break out; if (i0 > 0) i0 = 0; break;
    }
  }
  return i0 > 0 ? s.slice(0, i0) + s.slice(i1 + 1) : s;
}

function formatDate$1(date) {
  return format(date, "Invalid Date");
}

// Memoize the last-returned locale.
function localize(f) {
  let key = localize, value;
  return (locale = "en") => locale === key ? value : (value = f(key = locale));
}

function ascending(a, b) {
  return defined(b) - defined(a) || (a < b ? -1 : a > b ? 1 : a >= b ? 0 : NaN);
}

function descending(b, a) {
  return defined(a) - defined(b) || (a < b ? -1 : a > b ? 1 : a >= b ? 0 : NaN);
}

function defined(d) {
  return d != null && !Number.isNaN(d);
}

const first = ([x]) => x;
const second = ([, x]) => x;

function createChooser({multiple: fixedMultiple, render, selectedIndexes, select}) {
  return function chooser(data, {
    locale,
    keyof = data instanceof Map ? first : identity,
    valueof = data instanceof Map ? second : identity,
    format = (f => (d, i, data) => f(keyof(d, i, data)))(formatLocaleAuto(locale)),
    multiple,
    key,
    value,
    disabled = false,
    sort,
    unique,
    ...options
  } = {}) {
    if (typeof keyof !== "function") throw new TypeError("keyof is not a function");
    if (typeof valueof !== "function") throw new TypeError("valueof is not a function");
    if (typeof format !== "function") throw new TypeError("format is not a function");
    if (fixedMultiple !== undefined) multiple = fixedMultiple;
    sort = maybeSort(sort);
    let size = +multiple;
    if (value === undefined) value = key !== undefined && data instanceof Map ? (size > 0 ? Array.from(key, k => data.get(k)) : data.get(key)) : undefined;
    unique = !!unique;
    data = arrayify(data);
    let keys = data.map((d, i) => [keyof(d, i, data), i]);
    if (sort !== undefined) keys.sort(([a], [b]) => sort(a, b));
    if (unique) keys = [...new Map(keys.map(o => [intern(o[0]), o])).values()];
    const index = keys.map(second);
    if (multiple === true) size = Math.max(1, Math.min(10, index.length));
    else if (size > 0) multiple = true;
    else multiple = false, size = undefined;
    const [form, input] = render(
      data,
      index,
      maybeSelection(data, index, value, multiple, valueof),
      maybeDisabled(data, index, disabled, valueof),
      {
        ...options,
        format,
        multiple,
        size
      }
    );
    form.addEventListener("input", oninput);
    form.addEventListener("change", dispatchInput);
    form.addEventListener("submit", preventDefault);
    function oninput(event) {
      if (event && event.isTrusted) form.removeEventListener("change", dispatchInput);
      if (multiple) {
        value = selectedIndexes(input).map(i => valueof(data[i], i, data));
      } else {
        const i = selectedIndex(input);
        value = i < 0 ? null : valueof(data[i], i, data);
      }
    }
    oninput();
    return Object.defineProperty(form, "value", {
      get() {
        return value;
      },
      set(v) {
        if (multiple) {
          const selection = new Set(v);
          for (const e of input) {
            const i = +e.value;
            select(e, selection.has(valueof(data[i], i, data)));
          }
        } else {
          input.value = index.find(i => v === valueof(data[i], i, data));
        }
        oninput();
      }
    });
  };
}

function maybeSelection(data, index, value, multiple, valueof) {
  const values = new Set(value === undefined ? [] : multiple ? arrayify(value) : [value]);
  if (!values.size) return () => false;
  const selection = new Set();
  for (const i of index) {
    if (values.has(valueof(data[i], i, data))) {
      selection.add(i);
    }
  }
  return i => selection.has(i);
}

function maybeDisabled(data, index, value, valueof) {
  if (typeof value === "boolean") return value;
  const values = new Set(arrayify(value));
  const disabled = new Set();
  for (const i of index) {
    if (values.has(valueof(data[i], i, data))) {
      disabled.add(i);
    }
  }
  return i => disabled.has(i);
}

function maybeSort(sort) {
  if (sort === undefined || sort === false) return;
  if (sort === true || sort === "ascending") return ascending;
  if (sort === "descending") return descending;
  if (typeof sort === "function") return sort;
  throw new TypeError("sort is not a function");
}

function selectedIndex(input) {
  return input.value ? +input.value : -1;
}

function intern(value) {
  return value !== null && typeof value === "object" ? value.valueOf() : value;
}

function createCheckbox(multiple, type) {
  return createChooser({
    multiple,
    render(data, index, selected, disabled, {format, label}) {
      const form = html`<form class="inputs-3a86ea inputs-3a86ea-checkbox">
      ${maybeLabel(label)}<div>
        ${index.map(i => html`<label><input type=${type} disabled=${typeof disabled === "function" ? disabled(i) : disabled} name=input value=${i} checked=${selected(i)}>${format(data[i], i, data)}`)}
      </div>
    </form>`;
      return [form, inputof$1(form.elements.input, multiple)];
    },
    selectedIndexes(input) {
      return Array.from(input).filter(i => i.checked).map(i => +i.value);
    },
    select(input, selected) {
      input.checked = selected;
    }
  });
}

const radio = createCheckbox(false, "radio");

const checkbox = createCheckbox(true, "checkbox");

function toggle({label, value, values, disabled} = {}) {
  const input = html`<input class=inputs-3a86ea-input type=checkbox name=input disabled=${disabled}>`;
  const form = html`<form class="inputs-3a86ea inputs-3a86ea-toggle">${maybeLabel(label, input)}${input}`;
  Object.defineProperty(form, "value", {
    get() {
      return values === undefined ? input.checked : values[input.checked ? 0 : 1];
    },
    set(v) {
      input.checked = values === undefined ? !!v : v === values[0];
    }
  });
  if (value !== undefined) form.value = value;
  return form;
}

// The input is undefined if there are no options, or an individual input
// element if there is only one; we want these two cases to behave the same as
// when there are two or more options, i.e., a RadioNodeList.
function inputof$1(input, multiple) {
  return input === undefined ? new OptionZero(multiple ? [] : null)
    : typeof input.length === "undefined" ? new (multiple ? MultipleOptionOne : OptionOne)(input)
    : input;
}

class OptionZero {
  constructor(value) {
    this._value = value;
  }
  get value() {
    return this._value;
  }
  set value(v) {
    // ignore
  }
  *[Symbol.iterator]() {
    // empty
  }
}

// TODO If we allow selected radios to be cleared by command-clicking, then
// assigning a radio’s value programmatically should also clear the selection.
// This will require changing this class and also wrapping RadioNodeList in the
// common case to change the value setter’s behavior.
class OptionOne {
  constructor(input) {
    this._input = input;
  }
  get value() {
    const {_input} = this;
    return _input.checked ? _input.value : "";
  }
  set value(v) {
    const {_input} = this;
    if (_input.checked) return;
    _input.checked = stringify(v) === _input.value;
  }
  *[Symbol.iterator]() {
    yield this._input;
  }
}

class MultipleOptionOne {
  constructor(input) {
    this._input = input;
    this._value = input.checked ? [input.value] : [];
  }
  get value() {
    return this._value;
  }
  set value(v) {
    const {_input} = this;
    if (_input.checked) return;
    _input.checked = stringify(v) === _input.value;
    this._value = _input.checked ? [_input.value] : [];
  }
  *[Symbol.iterator]() {
    yield this._input;
  }
}

function maybeDatalist(datalist) {
  if (datalist === undefined) return [null, null];
  const listId = newId();
  const list = html`<datalist id=${listId}>${Array.from(datalist, value => html`<option value=${stringify(value)}>`)}`;
  return [list, listId];
}

function createText(form, input, value, {
  validate = checkValidity,
  submit
} = {}, {
  get = (input) => input.value,
  set = (input, value) => input.value = stringify(value),
  same = (input, value) => input.value === value,
  after = (button) => input.after(button)
} = {}) {
  submit = submit === true ? "Submit" : submit || null;
  const button = submit ? html`<button type=submit disabled>${submit}` : null;
  if (submit) after(button);
  set(input, value);
  value = validate(input) ? get(input) : undefined;
  form.addEventListener("submit", onsubmit);
  input.oninput = oninput;
  function update() {
    if (validate(input)) {
      value = get(input);
      return true;
    }
  }
  function onsubmit(event) {
    preventDefault(event);
    if (submit) {
      if (update()) {
        button.disabled = true;
        dispatchInput(event);
      } else {
        input.reportValidity();
      }
    }
  }
  function oninput(event) {
    if (submit) {
      button.disabled = same(input, value);
      event.stopPropagation();
    } else if (!update()) {
      event.stopPropagation();
    }
  }
  return Object.defineProperty(form, "value", {
    get() {
      return value;
    },
    set(v) {
      set(input, v);
      update();
    }
  });
}

function text({
  label,
  value = "",
  type = "text",
  placeholder,
  pattern,
  spellcheck,
  autocomplete,
  autocapitalize,
  min,
  max,
  minlength,
  maxlength,
  required = minlength > 0,
  datalist,
  readonly,
  disabled,
  width,
  ...options
} = {}) {
  const [list, listId] = maybeDatalist(datalist);
  const input = html`<input
    type=${type}
    name=text
    list=${listId}
    readonly=${readonly}
    disabled=${disabled}
    required=${required}
    min=${min}
    max=${max}
    minlength=${minlength}
    maxlength=${maxlength}
    pattern=${pattern}
    spellcheck=${truefalse(spellcheck)}
    autocomplete=${onoff(autocomplete)}
    autocapitalize=${onoff(autocapitalize)}
    placeholder=${placeholder}
  >`;
  const form = html`<form class=inputs-3a86ea style=${maybeWidth(width)}>
    ${maybeLabel(label, input)}<div class=inputs-3a86ea-input>
      ${input}
    </div>${list}
  </form>`;
  return createText(form, input, value, options);
}

function email(options) {
  return text({...options, type: "email"});
}

function tel(options) {
  return text({...options, type: "tel"});
}

function url(options) {
  return text({...options, type: "url"});
}

function password(options) {
  return text({...options, type: "password"});
}

// Hypertext Literal will normally drop an attribute if its value is exactly
// false, but for these attributes (e.g., spellcheck), we actually want the
// false to be stringified as the attribute value.
function truefalse(value) {
  return value == null ? null : `${value}`;
}

// For boolean attributes that support “on” and “off”, this maps true to “on”
// and false to “off”. Any other value (if not nullish) is assumed to be a
// string, such as autocapitalize=sentences.
function onoff(value) {
  return value == null ? null : `${value === false ? "off" : value === true ? "on" : value}`;
}

function color({
  label,
  value,
  required,
  datalist,
  readonly,
  disabled,
  width,
  ...options
} = {}) {
  const [list, listId] = maybeDatalist(datalist);
  const id = newId();
  const input = html`<input
    type=color
    name=text
    value=${value}
    id=${id}
    list=${listId}
    readonly=${readonly}
    disabled=${disabled}
    required=${required}
  >`;
  const output = html`<output
    for=${id}
  >`;
  output.value = input.value;
  input.addEventListener("input", () => output.value = input.value);
  const form = html`<form class=inputs-3a86ea style=${maybeWidth(width)}>
    ${maybeLabel(label, input)}<div class=inputs-3a86ea-input>
      <div class=inputs-3a86ea-input>${input}${output}</div>
    </div>${list}
  </form>`;
  return createText(form, input, value, options, {
    after: (button) => input.parentNode.after(button)
  });
}

const dateops = {
  type: "date",
  get: (input) => input.valueAsDate,
  set: (input, value) => input.value = formatDate(value),
  same: (input, value) => +input.valueAsDate === +value,
  format: formatDate
};

// datetime-local pedantically refuses to support valueAsDate, so here we use
// the Date constructor to convert the input based on the user’s local time zone
// (which you think would be implied by datetime-local)?
// https://github.com/whatwg/html/issues/4770
const datetimeops = {
  type: "datetime-local",
  get: (input) => input.value ? new Date(input.value) : null,
  set: (input, value) => input.value = formatDatetime(value),
  same: (input, value) => +new Date(input.value) === +value,
  format: formatDatetime
};

function createDate({
  label,
  min,
  max,
  required,
  readonly,
  disabled,
  width,
  value,
  ...options
} = {}, {
  type,
  format,
  ...ops
}) {
  const input = html`<input type=${type} name=date readonly=${readonly} disabled=${disabled} required=${required} min=${format(min)} max=${format(max)}>`;
  const form = html`<form class=inputs-3a86ea style=${maybeWidth(width)}>
    ${maybeLabel(label, input)}<div class=inputs-3a86ea-input>
      ${input}
    </div>
  </form>`;
  return createText(form, input, coerce(value), options, ops);
}

function date(options) {
  return createDate(options, dateops);
}

function datetime(options) {
  return createDate(options, datetimeops);
}

function coerce(value) {
  return value instanceof Date && !isNaN(value) ? value
    : typeof value === "string" ? parse(value, null)
    : value == null || isNaN(value = +value) ? null
    : new Date(+value);
}

function formatDate(value) {
  return (value = coerce(value))
    ? value.toISOString().slice(0, 10)
    : value;
}

// The datetime-local input uses YYYY-MM-DDThh:mm like ISO 8601, but in local
// time rather than UTC, so we apply the offset before calling toISOString.
// https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input/datetime-local
function formatDatetime(value) {
  return (value = coerce(value))
    ? (new Date(+value - value.getTimezoneOffset() * 1000 * 60)).toISOString().slice(0, 16)
    : value;
}

function form(inputs, options) {
  return (Array.isArray(inputs) ? arrayForm : objectForm)(inputs, options);
}

function arrayTemplate(inputs) {
  return html`<div>${inputs}`;
}

function arrayForm(inputs, {template = arrayTemplate} = {}) {
  inputs = [...inputs]; // defensive copy
  let value = inputs.map(({value}) => value);
  return Object.defineProperty(template(inputs), "value", {
    get() {
      for (let i = 0, n = inputs.length; i < n; ++i) {
        const v = inputs[i].value;
        if (!Object.is(v, value[i])) {
          value = [...value];
          value[i] = v;
        }
      }
      return value;
    },
    set(v = []) {
      for (let i = 0, n = inputs.length; i < n; ++i) {
        inputs[i].value = v[i];
      }
    }
  });
}

function objectTemplate(inputs) {
  return html`<div>${Object.values(inputs)}`;
}

function objectForm(inputs, {template = objectTemplate} = {}) {
  inputs = {...inputs}; // defensive copy
  let value = Object.fromEntries(Object.entries(inputs).map(([name, {value}]) => [name, value]));
  return Object.defineProperty(template(inputs), "value", {
    get() {
      for (const k in value) {
        const v = inputs[k].value;
        if (!Object.is(v, value[k])) {
          value = {...value};
          value[k] = v;
        }
      }
      return value;
    },
    set(v = {}) {
      for (const name in inputs) {
        inputs[name].value = v[name];
      }
    }
  });
}

function file({
  label,
  required,
  accept,
  capture,
  multiple,
  disabled,
  width,
  value, // eslint-disable-line no-unused-vars
  submit, // eslint-disable-line no-unused-vars
  transform = (file) => file,
  ...options
} = {}) {
  const input = html`<input
    type=file
    name=file
    disabled=${disabled}
    required=${required}
    accept=${accept}
    capture=${capture}
    multiple=${multiple}
  >`;
  const form = html`<form class=inputs-3a86ea style=${maybeWidth(width)}>
    ${maybeLabel(label, input)}<div class=inputs-3a86ea-input>
      ${input}
    </div>
  </form>`;
  return createText(form, input, undefined, options, {
    get: (input) => multiple ? Array.from(input.files, (file) => transform(file))
      : input.files.length ? transform(input.files[0])
      : null,
    set: () => {}, // ignored
    same: () => false // ignored
  });
}

const epsilon = 1e-6;

function number(extent, options) {
  if (arguments.length < 2) options = extent, extent = undefined;
  if (extent === undefined) extent = [];
  return createRange({extent}, options);
}

function range(extent = [0, 1], options) {
  return createRange({extent, range: true}, options);
}

function createRange({
  extent: [min, max],
  range
}, {
  format = formatTrim,
  transform,
  invert,
  label = "",
  value: initialValue,
  step,
  disabled,
  placeholder,
  validate = checkValidity,
  width
} = {}) {
  let value;
  if (typeof format !== "function") throw new TypeError("format is not a function");
  if (min == null || isNaN(min = +min)) min = -Infinity;
  if (max == null || isNaN(max = +max)) max = Infinity;
  if (min > max) [min, max] = [max, min], transform === undefined && (transform = negate);
  if (step !== undefined) step = +step;
  const number = html`<input type=number min=${isFinite(min) ? min : null} max=${isFinite(max) ? max : null} step=${step == undefined ? "any" : step} name=number required placeholder=${placeholder} oninput=${onnumber} disabled=${disabled}>`;
  let irange; // untransformed range for coercion
  if (range) {
    if (transform === undefined) transform = identity;
    if (typeof transform !== "function") throw new TypeError("transform is not a function");
    if (invert === undefined) invert = transform.invert === undefined ? solver(transform) : transform.invert;
    if (typeof invert !== "function") throw new TypeError("invert is not a function");
    let tmin = +transform(min), tmax = +transform(max);
    if (tmin > tmax) [tmin, tmax] = [tmax, tmin];
    range = html`<input type=range min=${isFinite(tmin) ? tmin : null} max=${isFinite(tmax) ? tmax : null} step=${step === undefined || (transform !== identity && transform !== negate) ? "any" : step} name=range oninput=${onrange} disabled=${disabled}>`;
    irange = transform === identity ? range : html`<input type=range min=${min} max=${max} step=${step === undefined ? "any" : step} name=range disabled=${disabled}>`;
  } else {
    range = null;
    transform = invert = identity;
  }
  const form = html`<form class=inputs-3a86ea style=${maybeWidth(width)}>
    ${maybeLabel(label, number)}<div class=inputs-3a86ea-input>
      ${number}${range}
    </div>
  </form>`;
  form.addEventListener("submit", preventDefault);
  // If range, use an untransformed range to round to the nearest valid value.
  function coerce(v) {
    if (!irange) return +v;
    v = Math.max(min, Math.min(max, v));
    if (!isFinite(v)) return v;
    irange.valueAsNumber = v;
    return irange.valueAsNumber;
  }
  function onrange(event) {
    const v = coerce(invert(range.valueAsNumber));
    if (isFinite(v)) {
      number.valueAsNumber = Math.max(min, Math.min(max, v));
      if (validate(number)) {
        value = number.valueAsNumber;
        number.value = format(value);
        return;
      }
    }
    if (event) event.stopPropagation();
  }
  function onnumber(event) {
    const v = coerce(number.valueAsNumber);
    if (isFinite(v)) {
      if (range) range.valueAsNumber = transform(v);
      if (validate(number)) {
        value = v;
        return;
      }
    }
    if (event) event.stopPropagation();
  }
  Object.defineProperty(form, "value", {
    get() {
      return value;
    },
    set(v) {
      v = coerce(v);
      if (isFinite(v)) {
        number.valueAsNumber = v;
        if (range) range.valueAsNumber = transform(v);
        if (validate(number)) {
          value = v;
          number.value = format(value);
        }
      }
    }
  });
  if (initialValue === undefined && irange) initialValue = irange.valueAsNumber; // (min + max) / 2
  if (initialValue !== undefined) form.value = initialValue; // invoke setter
  return form;
}

function negate(x) {
  return -x;
}

function square(x) {
  return x * x;
}

function solver(f) {
  if (f === identity || f === negate) return f;
  if (f === Math.sqrt) return square;
  if (f === Math.log) return Math.exp;
  if (f === Math.exp) return Math.log;
  return x => solve(f, x, x);
}

function solve(f, y, x) {
  let steps = 100, delta, f0, f1;
  x = x === undefined ? 0 : +x;
  y = +y;
  do {
    f0 = f(x);
    f1 = f(x + epsilon);
    if (f0 === f1) f1 = f0 + epsilon;
    x -= delta = (-1 * epsilon * (f0 - y)) / (f0 - f1);
  } while (steps-- > 0 && Math.abs(delta) > epsilon);
  return steps < 0 ? NaN : x;
}

function search(data, {
  locale,
  format = formatResults(locale), // length format
  label,
  query = "", // initial search query
  placeholder = "Search", // placeholder text to show when empty
  columns = maybeColumns(data),
  spellcheck,
  autocomplete,
  autocapitalize,
  filter = columns === undefined ? searchFilter : columnFilter(columns), // returns the filter function given query
  datalist,
  disabled,
  required = true, // if true, the value is everything if nothing is selected
  width
} = {}) {
  let value = [];
  data = arrayify(data);
  required = !!required;
  const [list, listId] = maybeDatalist(datalist);
  const input = html`<input
    name=input
    type=search
    list=${listId}
    disabled=${disabled}
    spellcheck=${truefalse(spellcheck)}
    autocomplete=${onoff(autocomplete)}
    autocapitalize=${onoff(autocapitalize)}
    placeholder=${placeholder}
    value=${query}
    oninput=${oninput}
  >`;
  const output = html`<output name=output>`;
  const form = html`<form class=inputs-3a86ea style=${maybeWidth(width)}>
    ${maybeLabel(label, input)}<div class=inputs-3a86ea-input>
      ${input}${output}
    </div>${list}
  </form>`;
  form.addEventListener("submit", preventDefault);
  function oninput() {
    value = input.value || required ? data.filter(filter(input.value)) : [];
    if (columns !== undefined) value.columns = columns;
    output.value = format(value.length);
  }
  oninput();
  return Object.defineProperties(form, {
    value: {
      get() {
        return value;
      }
    },
    query: {
      get() {
        return query;
      },
      set(v) {
        query = input.value = stringify(v);
        oninput();
      }
    }
  });
}

function searchFilter(query) {
  const filters = `${query}`.split(/\s+/g).filter(t => t).map(termFilter);
  return d => {
    if (d == null) return false;
    if (typeof d === "object") {
      out: for (const filter of filters) {
        for (const value of valuesof(d)) {
          if (filter.test(value)) {
            continue out;
          }
        }
        return false;
      }
    } else {
      for (const filter of filters) {
        if (!filter.test(d)) {
          return false;
        }
      }
    }
    return true;
  };
}

function columnFilter(columns) {
  return query => {
    const filters = `${query}`.split(/\s+/g).filter(t => t).map(termFilter);
    return d => {
      out: for (const filter of filters) {
        for (const column of columns) {
          if (filter.test(d[column])) {
            continue out;
          }
        }
        return false;
      }
      return true;
    };
  };
}

function* valuesof(d) {
  for (const key in d) {
    yield d[key];
  }
}

function termFilter(term) {
  return new RegExp(`(?:^|[^\\p{L}-])${escapeRegExp(term)}`, "iu");
}

function escapeRegExp(text) {
  return text.replace(/[\\^$.*+?()[\]{}|]/g, "\\$&");
}

const formatResults = localize(locale => {
  const formatNumber = formatLocaleNumber(locale);
  return length => `${formatNumber(length)} result${length === 1 ? "" : "s"}`;
});

const select = createChooser({
  render(data, index, selected, disabled, {format, multiple, size, label, width}) {
    const select = html`<select class=inputs-3a86ea-input disabled=${disabled === true} multiple=${multiple} size=${size} name=input>
      ${index.map(i => html`<option value=${i} disabled=${typeof disabled === "function" ? disabled(i) : false} selected=${selected(i)}>${stringify(format(data[i], i, data))}`)}
    </select>`;
    const form = html`<form class=inputs-3a86ea style=${maybeWidth(width)}>${maybeLabel(label, select)}${select}`;
    return [form, select];
  },
  selectedIndexes(input) {
    return Array.from(input.selectedOptions, i => +i.value);
  },
  select(input, selected) {
    input.selected = selected;
  }
});

const rowHeight = 22;

function table(data, options = {}) {
  const {
    rows = 11.5, // maximum number of rows to show
    height,
    maxHeight = height === undefined ? (rows + 1) * rowHeight - 1 : undefined,
    width = {}, // object of column name to width, or overall table width
    maxWidth
  } = options;
  const id = newId();
  const root = html`<form class="inputs-3a86ea inputs-3a86ea-table" id=${id} style=${{height: length(height), maxHeight: length(maxHeight), width: typeof width === "string" || typeof width === "number" ? length(width) : undefined, maxWidth: length(maxWidth)}}>`;
  // The outer form element is created synchronously, while the table is lazily
  // created when the data promise resolves. This allows you to pass a promise
  // of data to the table without an explicit await.
  if (data && typeof data.then === "function") {
    Object.defineProperty(root, "value", {
      configurable: true, // allow defineProperty again on initialization
      set() {
        throw new Error("cannot set value while data is unresolved");
      }
    });
    Promise.resolve(data).then(data => initialize({root, id}, data, options));
  } else {
    initialize({root, id}, data, options);
  }
  return root;
}

function initialize(
  {
    root,
    id
  },
  data,
  {
    columns = maybeColumns(data), // array of column names
    value, // initial selection
    required = true, // if true, the value is everything if nothing is selected
    sort, // name of column to sort by, if any
    reverse = false, // if sorting, true for descending and false for ascending
    format, // object of column name to format function
    locale,
    align, // object of column name to left, right, or center
    header, // object of column name to string or HTML element
    rows = 11.5, // maximum number of rows to show
    width = {}, // object of column name to width, or overall table width
    multiple = true,
    select: selectable = true, // is the table selectable?
    layout // "fixed" or "auto"
  } = {}
) {
  columns = columns === undefined ? columnsof(data) : arrayify(columns);
  if (layout === undefined) layout = columns.length >= 12 ? "auto" : "fixed";
  format = formatof(format, data, columns, locale);
  align = alignof(align, data, columns);

  let array = [];
  let index = [];
  let iterator = data[Symbol.iterator]();
  let iterindex = 0;
  let N = lengthof(data); // total number of rows (if known)
  let n = minlengthof(rows * 2); // number of currently-shown rows

  // Defer materializing index and data arrays until needed.
  function materialize() {
    if (iterindex >= 0) {
      iterindex = iterator = undefined;
      index = Uint32Array.from(array = arrayify(data), (_, i) => i);
      N = index.length;
    }
  }

  function minlengthof(length) {
    length = Math.floor(length);
    if (N !== undefined) return Math.min(N, length);
    if (length <= iterindex) return length;
    while (length > iterindex) {
      const {done, value} = iterator.next();
      if (done) return N = iterindex;
      index.push(iterindex++);
      array.push(value);
    }
    return iterindex;
  }

  let currentSortHeader = null, currentReverse = false;
  let selected = new Set();
  let anchor = null, head = null;

  const tbody = html`<tbody>`;
  const tr = html`<tr><td>${selectable ? html`<input type=${multiple ? "checkbox" : "radio"} name=${multiple ? null : "radio"}>` : null}</td>${columns.map(() => html`<td>`)}`;
  const theadr = html`<tr><th>${selectable ? html`<input type=checkbox onclick=${reselectAll} disabled=${!multiple}>` : null}</th>${columns.map((column) => html`<th title=${column} onclick=${event => resort(event, column)}><span></span>${header && column in header ? header[column] : column}</th>`)}</tr>`;
  root.appendChild(html.fragment`<table style=${{tableLayout: layout}}>
  <thead>${minlengthof(1) || columns.length ? theadr : null}</thead>
  ${tbody}
</table>
<style>${columns.map((column, i) => {
  const rules = [];
  if (align[column] != null) rules.push(`text-align:${align[column]}`);
  if (width[column] != null) rules.push(`width:${length(width[column])}`);
  if (rules.length) return `#${id} tr>:nth-child(${i + 2}){${rules.join(";")}}`;
}).filter(identity).join("\n")}</style>`);
  function appendRows(i, j) {
    if (iterindex === i) {
      for (; i < j; ++i) {
        appendRow(iterator.next().value, i);
      }
      iterindex = j;
    } else {
      for (let k; i < j; ++i) {
        k = index[i];
        appendRow(array[k], k);
      }
    }
  }

  function appendRow(d, i) {
    const itr = tr.cloneNode(true);
    const input = inputof(itr);
    if (input != null) {
      input.onclick = reselect;
      input.checked = selected.has(i);
      input.value = i;
    }
    if (d != null) for (let j = 0; j < columns.length; ++j) {
      let column = columns[j];
      let value = d[column];
      if (!defined(value)) continue;
      value = format[column](value, i, data);
      if (!(value instanceof Node)) value = document.createTextNode(value);
      itr.childNodes[j + 1].appendChild(value);
    }
    tbody.append(itr);
  }

  function unselect(i) {
    materialize();
    let j = index.indexOf(i);
    if (j < tbody.childNodes.length) {
      const tr = tbody.childNodes[j];
      inputof(tr).checked = false;
    }
    selected.delete(i);
  }

  function select(i) {
    materialize();
    let j = index.indexOf(i);
    if (j < tbody.childNodes.length) {
      const tr = tbody.childNodes[j];
      inputof(tr).checked = true;
    }
    selected.add(i);
  }

  function* range(i, j) {
    materialize();
    i = index.indexOf(i), j = index.indexOf(j);
    if (i < j) while (i <= j) yield index[i++];
    else while (j <= i) yield index[j++];
  }

  function first(set) {
    return set[Symbol.iterator]().next().value;
  }

  function reselectAll(event) {
    materialize();
    if (this.checked) {
      selected = new Set(index);
      for (const tr of tbody.childNodes) {
        inputof(tr).checked = true;
      }
    } else {
      for (let i of selected) unselect(i);
      anchor = head = null;
      if (event.detail) event.currentTarget.blur();
    }
    reinput();
  }

  function reselect(event) {
    materialize();
    let i = +this.value;
    if (!multiple) {
      for (let i of selected) unselect(i);
      select(i);
    } else if (event.shiftKey) {
      if (anchor === null) anchor = selected.size ? first(selected) : index[0];
      else for (let i of range(anchor, head)) unselect(i);
      head = i;
      for (let i of range(anchor, head)) select(i);
    } else {
      anchor = head = i;
      if (selected.has(i)) {
        unselect(i);
        anchor = head = null;
        if (event.detail) event.currentTarget.blur();
      } else {
        select(i);
      }
    }
    reinput();
  }

  function resort(event, column) {
    materialize();
    const th = event.currentTarget;
    let compare;
    if (currentSortHeader === th && event.metaKey) {
      orderof(currentSortHeader).textContent = "";
      currentSortHeader = null;
      currentReverse = false;
      compare = ascending;
    } else {
      if (currentSortHeader === th) {
        currentReverse = !currentReverse;
      } else {
        if (currentSortHeader) {
          orderof(currentSortHeader).textContent = "";
        }
        currentSortHeader = th;
        currentReverse = event.altKey;
      }
      const order = currentReverse ? descending : ascending;
      compare = (a, b) => order(array[a][column], array[b][column]);
      orderof(th).textContent = currentReverse ? "▾"  : "▴";
    }
    index.sort(compare);
    selected = new Set(Array.from(selected).sort(compare));
    root.scrollTo(root.scrollLeft, 0);
    while (tbody.firstChild) tbody.firstChild.remove();
    appendRows(0, n = minlengthof(rows * 2));
    anchor = head = null;
    reinput();
  }

  function reinput() {
    const check = inputof(theadr);
    if (check == null) return;
    check.disabled = !multiple && !selected.size;
    check.indeterminate = multiple && selected.size && selected.size !== N; // assume materalized!
    check.checked = selected.size;
    value = undefined; // lazily computed
  }

  root.addEventListener("scroll", () => {
    if (root.scrollHeight - root.scrollTop < rows * rowHeight * 1.5 && n < minlengthof(n + 1)) {
      appendRows(n, n = minlengthof(n + rows));
    }
  });

  if (sort === undefined && reverse) {
    materialize();
    index.reverse();
  }

  if (value !== undefined) {
    materialize();
    if (multiple) {
      const values = new Set(value);
      selected = new Set(index.filter(i => values.has(array[i])));
    } else {
      const i = array.indexOf(value);
      selected = i < 0 ? new Set() : new Set([i]);
    }
    reinput();
  }

  if (minlengthof(1)) {
    appendRows(0, n);
  } else {
    tbody.append(html`<tr>${columns.length ? html`<td>` : null}<td rowspan=${columns.length} style="padding-left: var(--length3); font-style: italic;">No results.</td></tr>`);
  }

  if (sort !== undefined) {
    let i = columns.indexOf(sort);
    if (i >= 0) {
      if (reverse) currentSortHeader = theadr.childNodes[i + 1];
      resort({currentTarget: theadr.childNodes[i + 1]}, columns[i]);
    }
  }

  return Object.defineProperty(root, "value", {
    get() {
      if (value === undefined) {
        materialize();
        if (multiple) {
          value = Array.from(required && selected.size === 0 ? index : selected, i => array[i]);
          value.columns = columns;
        } else if (selected.size) {
          const [i] = selected;
          value = array[i];
        } else {
          value = null;
        }
      }
      return value;
    },
    set(v) {
      materialize();
      if (multiple) {
        const values = new Set(v);
        const selection = new Set(index.filter(i => values.has(array[i])));
        for (const i of selected) if (!selection.has(i)) unselect(i);
        for (const i of selection) if (!selected.has(i)) select(i);
      } else {
        const i = array.indexOf(v);
        selected = i < 0 ? new Set() : new Set([i]);
      }
      value = undefined; // lazily computed
    }
  });
}

function inputof(tr) {
  return tr.firstChild.firstChild;
}

function orderof(th) {
  return th.firstChild;
}

function formatof(base = {}, data, columns, locale) {
  const format = Object.create(null);
  for (const column of columns) {
    if (column in base) {
      format[column] = base[column];
      continue;
    }
    switch (type(data, column)) {
      case "number": format[column] = formatLocaleNumber(locale); break;
      case "date": format[column] = formatDate$1; break;
      default: format[column] = formatLocaleAuto(locale); break;
    }
  }
  return format;
}

function alignof(base = {}, data, columns) {
  const align = Object.create(null);
  for (const column of columns) {
    if (column in base) {
      align[column] = base[column];
    } else if (type(data, column) === "number") {
      align[column] = "right";
    }
  }
  return align;
}

function type(data, column) {
  if (isArrowTable(data)) return getArrowType(data, column);
  for (const d of data) {
    if (d == null) continue;
    const value = d[column];
    if (value == null) continue;
    if (typeof value === "number") return "number";
    if (value instanceof Date) return "date";
    return;
  }
}

// https://github.com/observablehq/stdlib/blob/746ca2e69135df6178e4f3a17244def35d8d6b20/src/arrow.js#L4-L16
function isArrowTable(value) {
  return (
    typeof value.getChild === "function" &&
    typeof value.toArray === "function" &&
    value.schema &&
    Array.isArray(value.schema.fields)
  );
}

// https://github.com/apache/arrow/blob/89f9a0948961f6e94f1ef5e4f310b707d22a3c11/js/src/enum.ts#L140-L141
function getArrowType(value, column) {
  const field = value.schema.fields.find((d) => d.name === column);
  switch (field?.type.typeId) {
    case 8: // Date
    case 10: // Timestamp
      return field.type.unit === 1 ? "date" : "number"; // millisecond
    case 2: // Int
    case 3: // Float
    case 7: // Decimal
    case 9: // Time
      return "number";
  }
}

function lengthof(data) {
  if (typeof data.length === "number") return data.length; // array or array-like
  if (typeof data.size === "number") return data.size; // map, set
  if (typeof data.numRows === "function") return data.numRows(); // arquero
}

function columnsof(data) {
  const columns = new Set();
  for (const row of data) {
    for (const name in row) {
      columns.add(name);
    }
  }
  return Array.from(columns);
}

function textarea({
  value = "",
  label,
  placeholder,
  spellcheck,
  autocomplete,
  autocapitalize,
  rows = 3,
  minlength,
  maxlength,
  required = minlength > 0,
  readonly,
  disabled,
  monospace = false,
  resize = rows < 12,
  width,
  ...options
} = {}) {
  const input = html`<textarea
    name=text
    readonly=${readonly}
    disabled=${disabled}
    required=${required}
    rows=${rows}
    minlength=${minlength}
    maxlength=${maxlength}
    spellcheck=${truefalse(spellcheck)}
    autocomplete=${onoff(autocomplete)}
    autocapitalize=${onoff(autocapitalize)}
    placeholder=${placeholder}
    onkeydown=${onkeydown}
    style=${{
      width,
      fontFamily: monospace ? "var(--monospace, monospace)" : null,
      resize: resize ? null : "none"
    }}
  >`;
  const form = html`<form class="inputs-3a86ea inputs-3a86ea-textarea" style=${maybeWidth(width)}>
    ${maybeLabel(label, input)}<div>
      ${input}
    </div>
  </form>`;
  function onkeydown(event) {
    if (options.submit && event.key === "Enter" && (event.metaKey || event.ctrlKey)) {
      return form.dispatchEvent(new Event("submit", bubbles));
    }
  }
  return createText(form, input, value, options);
}

function input(value) {
  const target = new EventTarget();
  target.value = value;
  return target;
}

function disposal(element) {
  return new Promise(resolve => {
    requestAnimationFrame(() => {
      const target = element.closest(".observablehq");
      if (!target) return resolve();
      const observer = new MutationObserver(() => {
        if (target.contains(element)) return;
        observer.disconnect(), resolve();
      });
      observer.observe(target, {childList: true});
    });
  });
}

function bind(target, source, invalidation = disposal(target)) {
  const sourceEvent = eventof(source);
  const onsource = () => set(target, source);
  const ontarget = () => (set(source, target), source.dispatchEvent(new Event(sourceEvent, bubbles)));
  onsource();
  target.addEventListener(eventof(target), ontarget);
  source.addEventListener(sourceEvent, onsource);
  invalidation.then(() => source.removeEventListener(sourceEvent, onsource));
  return target;
}

function get(input) {
  switch (input.type) {
    case "range":
    case "number": return input.valueAsNumber;
    case "date": return input.valueAsDate;
    case "checkbox": return input.checked;
    case "file": return input.multiple ? input.files : input.files[0];
    default: return input.value;
  }
}

function set(target, source) {
  const value = get(source);
  switch (target.type) {
    case "range":
    case "number": target.valueAsNumber = value; break;
    case "date": target.valueAsDate = value; break;
    case "checkbox": target.checked = value; break;
    case "file": target.multiple ? (target.files = value) : (target.files = [value]); break;
    default: target.value = value; break;
  }
}

function eventof(input) {
  switch (input.type) {
    case "button":
    case "submit": return "click";
    case "file": return "change";
    default: return "input";
  }
}

export { bind, button, checkbox, color, date, datetime, disposal, email, file, form, formatAuto, formatDate$1 as formatDate, formatLocaleAuto, formatLocaleNumber, formatNumber, formatTrim, input, number, password, radio, range, search, searchFilter, select, table, tel, text, textarea, toggle, url };
