#!/usr/bin/env node
import { join } from "node:path/posix";
import { parseArgs } from "node:util";
import * as clack from "@clack/prompts";
import wrapAnsi from "wrap-ansi";
import { readConfig } from "../config.js";
import { CliError } from "../error.js";
import { faint, link, red } from "../tty.js";
const args = process.argv.slice(2);
const CONFIG_OPTION = {
  root: {
    type: "string",
    description: "Path to the project root"
  },
  config: {
    type: "string",
    short: "c",
    description: "Path to the app config file"
  }
};
if (args[0] === "--version" || args[0] === "-v")
  args[0] = "version";
const { values, positionals, tokens } = parseArgs({
  options: {
    help: {
      type: "boolean",
      short: "h"
    },
    debug: {
      type: "boolean",
      short: "d"
    }
  },
  strict: false,
  tokens: true,
  args
});
let command;
if (positionals.length > 0) {
  const t = tokens.find((t2) => t2.kind === "positional");
  args.splice(t.index, 1);
  command = positionals[0];
  if (command === "help" && positionals.length > 1) {
    const p = tokens.find((p2) => p2.kind === "positional" && p2 !== t);
    args.splice(p.index - 1, 1, "--help");
    command = positionals[1];
  }
} else if (values.help) {
  const t = tokens.find((t2) => t2.kind === "option" && t2.name === "help");
  args.splice(t.index, 1);
  command = "help";
}
const CLACKIFIED_COMMANDS = ["create", "deploy", "login", "convert"];
try {
  switch (command) {
    case void 0:
    case "help": {
      helpArgs(command, { allowPositionals: true });
      console.log(
        `usage: observable <command>
  create       create a new app from a template
  preview      start the preview server
  build        generate a static site
  login        sign-in to Observable
  logout       sign-out of Observable
  deploy       deploy an app to Observable [deprecated]
  whoami       check authentication status
  convert      convert an Observable notebook to Markdown
  help         print usage information
  version      print the version`
      );
      if (command === void 0)
        process.exit(1);
      break;
    }
    case "version": {
      helpArgs(command, {});
      console.log("1.13.3");
      break;
    }
    case "build": {
      const {
        values: { config, root }
      } = helpArgs(command, {
        options: { ...CONFIG_OPTION }
      });
      await import("../build.js").then(async (build) => build.build({ config: await readConfig(config, root) }));
      break;
    }
    case "create": {
      helpArgs(command, {});
      await import("../create.js").then(async (create) => create.create());
      break;
    }
    case "deploy": {
      const {
        values: { config, root, message, build, id, "deploy-config": deployConfigPath }
      } = helpArgs(command, {
        options: {
          ...CONFIG_OPTION,
          message: {
            type: "string",
            short: "m",
            description: "Message to associate with this deploy"
          },
          build: {
            type: "boolean",
            description: "Always build before deploying"
          },
          "no-build": {
            type: "boolean",
            description: "Don\u2019t build before deploying; deploy as is"
          },
          id: {
            type: "string",
            hidden: true
          },
          "deploy-config": {
            type: "string",
            description: "Path to the deploy config file (deploy.json)"
          }
        }
      });
      await import("../deploy.js").then(
        async (deploy) => deploy.deploy({
          config: await readConfig(config, root),
          message,
          force: build === true ? "build" : build === false ? "deploy" : null,
          deployId: id,
          deployConfigPath
        })
      );
      break;
    }
    case "preview": {
      const { values: values2 } = helpArgs(command, {
        options: {
          ...CONFIG_OPTION,
          host: {
            type: "string",
            default: "127.0.0.1",
            description: "the server host; use 0.0.0.0 to accept external connections"
          },
          port: {
            type: "string",
            description: "the server port; defaults to 3000 (or higher if unavailable)"
          },
          cors: {
            type: "boolean",
            description: "allow cross-origin requests on all origins (*)"
          },
          "allow-origin": {
            type: "string",
            multiple: true,
            description: "allow cross-origin requests on a specific origin"
          },
          open: {
            type: "boolean",
            default: true,
            description: "open browser"
          },
          "no-open": {
            type: "boolean"
          }
        }
      });
      const { config, root, host, port, open, cors, ["allow-origin"]: origins } = values2;
      await readConfig(config, root);
      await import("../preview.js").then(
        async (preview) => preview.preview({
          config,
          root,
          hostname: host,
          port: port === void 0 ? void 0 : +port,
          origins: cors ? ["*"] : origins,
          open
        })
      );
      break;
    }
    case "login": {
      helpArgs(command, {});
      await import("../observableApiAuth.js").then((auth) => auth.login());
      break;
    }
    case "logout": {
      helpArgs(command, {});
      await import("../observableApiAuth.js").then((auth) => auth.logout());
      break;
    }
    case "whoami": {
      helpArgs(command, {});
      await import("../observableApiAuth.js").then((auth) => auth.whoami());
      break;
    }
    case "convert": {
      const {
        positionals: positionals2,
        values: { config, root, output: out, force }
      } = helpArgs(command, {
        options: {
          output: {
            type: "string",
            short: "o",
            description: "Output directory (defaults to the source root)"
          },
          force: {
            type: "boolean",
            short: "f",
            description: "If true, overwrite existing resources"
          },
          ...CONFIG_OPTION
        },
        allowPositionals: true
      });
      const output = out ?? join(root ?? ".", (await readConfig(config, root)).root);
      await import("../convert.js").then((convert) => convert.convert(positionals2, { output, force }));
      break;
    }
    default: {
      console.error(`observable: unknown command '${command}'. See 'observable help'.`);
      process.exit(1);
      break;
    }
  }
} catch (error) {
  const wrapWidth = Math.min(80, process.stdout.columns ?? 80);
  const bugMessage = "If you think this is a bug, please file an issue at";
  const bugUrl = "https://github.com/observablehq/framework/issues";
  const clackBugMessage = () => {
    console.log(`${faint("\u2502\n\u2502")}  ${bugMessage}
${faint("\u2514")}  ${link(bugUrl)}
`);
  };
  const consoleBugMessage = () => {
    console.error(`${bugMessage}
\u21B3 ${link(bugUrl)}
`);
  };
  if (error instanceof CliError) {
    if (error.print) {
      if (command && CLACKIFIED_COMMANDS.includes(command)) {
        clack.log.error(wrapAnsi(red(`Error: ${error.message}`), wrapWidth));
        clackBugMessage();
      } else {
        console.error(red(error.message));
        consoleBugMessage();
      }
    }
    process.exit(error.exitCode);
  } else {
    if (command && CLACKIFIED_COMMANDS.includes(command)) {
      clack.log.error(wrapAnsi(`${red("Error:")} ${error.message}`, wrapWidth));
      if (values.debug) {
        clack.outro("The full error follows");
        throw error;
      } else {
        clack.log.info("To see the full stack trace, run with the --debug flag.");
        clackBugMessage();
      }
    } else {
      console.error(`
${red("Unexpected error:")} ${error.message}`);
      if (values.debug) {
        console.error("The full error follows\n");
        throw error;
      } else {
        console.error("\nTip: To see the full stack trace, run with the --debug flag.\n");
        consoleBugMessage();
      }
    }
  }
  process.exit(1);
}
function helpArgs(command2, config) {
  const { options = {} } = config;
  const booleanPairs = [];
  for (const key in options) {
    if (options[key].type === "boolean" && !key.startsWith("no-") && options[`no-${key}`]?.type === "boolean") {
      booleanPairs.push(key);
    }
  }
  let result;
  try {
    result = parseArgs({
      ...config,
      tokens: config.tokens || booleanPairs.length > 0,
      options: { ...options, help: { type: "boolean", short: "h" }, debug: { type: "boolean" } },
      args
    });
  } catch (error) {
    if (!error.code?.startsWith("ERR_PARSE_ARGS_"))
      throw error;
    console.error(`observable: ${error.message}. See 'observable help${command2 ? ` ${command2}` : ""}'.`);
    process.exit(1);
  }
  if (result.values.help) {
    const publicOptions = Object.fromEntries(Object.entries(options).filter(([, option]) => !option.hidden));
    console.log(
      `Usage: observable ${command2}${command2 === void 0 || command2 === "help" ? " <command>" : ""}${Object.entries(
        publicOptions
      ).map(([name, { default: def }]) => ` [--${name}${def === void 0 ? "" : `=${def}`}]`).join("")}`
    );
    if (Object.values(publicOptions).some((spec) => spec.description)) {
      console.log();
      for (const [long, spec] of Object.entries(publicOptions)) {
        if (spec.description) {
          const left = `  ${spec.short ? `-${spec.short}, ` : ""}--${long}`.padEnd(20);
          console.log(`${left}${spec.description}`);
        }
      }
      console.log();
    }
    process.exit(0);
  }
  if ("tokens" in result && result.tokens) {
    const { values: values2, tokens: tokens2 } = result;
    for (const key of booleanPairs) {
      for (const token of tokens2) {
        if (token.kind !== "option")
          continue;
        const { name } = token;
        if (name === `no-${key}`)
          values2[key] = false;
        else if (name === key)
          values2[key] = true;
      }
    }
  }
  return result;
}
