import { createHash } from "node:crypto";
import { existsSync } from "node:fs";
import { copyFile, readFile, rm, stat, writeFile } from "node:fs/promises";
import { basename, dirname, extname, join } from "node:path/posix";
import { getDuckDBManifest } from "./duckdb.js";
import { CliError, enoent } from "./error.js";
import { getClientPath, prepareOutput } from "./files.js";
import { findModule, getModuleHash, readJavaScript } from "./javascript/module.js";
import { transpileModule } from "./javascript/transpile.js";
import { populateNpmCache, resolveNpmImport, rewriteNpmImports } from "./npm.js";
import { isAssetPath, isPathImport, relativePath, resolvePath, within } from "./path.js";
import { renderModule, renderPage } from "./render.js";
import { getModuleResolvers, getResolvers } from "./resolvers.js";
import { resolveStylesheetPath } from "./resolvers.js";
import { bundleStyles, rollupClient } from "./rollup.js";
import { searchIndex } from "./search.js";
import { Telemetry } from "./telemetry.js";
import { tree } from "./tree.js";
import { faint, green, magenta, red, yellow } from "./tty.js";
async function build({ config }, effects = new FileBuildEffects(config.output, join(config.root, ".observablehq", "cache"))) {
  const { root, loaders, title, duckdb } = config;
  Telemetry.record({ event: "build", step: "start" });
  await effects.prepare();
  const outputs = /* @__PURE__ */ new Map();
  const files = /* @__PURE__ */ new Set();
  const localImports = /* @__PURE__ */ new Set();
  const globalImports = /* @__PURE__ */ new Set();
  const stylesheets = /* @__PURE__ */ new Set();
  const addFile = (path, f) => files.add(resolvePath(path, f));
  const addLocalImport = (path, i) => localImports.add(resolvePath(path, i));
  const addGlobalImport = (path, i) => isPathImport(i) && globalImports.add(resolvePath(path, i));
  const addStylesheet = (path, s) => stylesheets.add(/^\w+:/.test(s) ? s : resolvePath(path, s));
  let assetCount = 0;
  let pageCount = 0;
  const pagePaths = /* @__PURE__ */ new Set();
  const buildManifest = {
    ...title && { title },
    config: { root },
    pages: [],
    modules: [],
    files: []
  };
  const addToManifest = (type, file, { title: title2, path }) => {
    buildManifest[type].push({
      path: config.normalizePath(file),
      source: join("/", path),
      // TODO have route return path with leading slash?
      ...title2 != null && { title: title2 }
    });
  };
  for await (const path of config.paths()) {
    effects.output.write(`${faint("load")} ${path} `);
    const start = performance.now();
    const options = { path, ...config };
    if (path.endsWith(".js")) {
      const module = findModule(root, path);
      if (module) {
        const resolvers2 = await getModuleResolvers(path, config);
        const elapsed2 = Math.floor(performance.now() - start);
        for (const f of resolvers2.files)
          addFile(path, f);
        for (const i of resolvers2.localImports)
          addLocalImport(path, i);
        for (const i of resolvers2.globalImports)
          addGlobalImport(path, resolvers2.resolveImport(i));
        for (const s of resolvers2.stylesheets)
          addStylesheet(path, s);
        effects.output.write(`${faint("in")} ${(elapsed2 >= 100 ? yellow : faint)(`${elapsed2}ms`)}
`);
        outputs.set(path, { type: "module", resolvers: resolvers2 });
        ++assetCount;
        addToManifest("modules", path, module);
        continue;
      }
    }
    const file = loaders.find(path);
    if (file) {
      effects.output.write(`${faint("copy")} ${join(root, path)} ${faint("\u2192")} `);
      const sourcePath = join(root, await file.load({ useStale: true }, effects));
      await effects.copyFile(sourcePath, path);
      addToManifest("files", path, file);
      ++assetCount;
      continue;
    }
    const page = await loaders.loadPage(path, options, effects);
    if (page.data.draft) {
      effects.logger.log(faint("(skipped)"));
      continue;
    }
    const resolvers = await getResolvers(page, options);
    const elapsed = Math.floor(performance.now() - start);
    for (const f of resolvers.assets)
      addFile(path, f);
    for (const f of resolvers.files)
      addFile(path, f);
    for (const i of resolvers.localImports)
      addLocalImport(path, i);
    for (const i of resolvers.globalImports)
      addGlobalImport(path, resolvers.resolveImport(i));
    for (const s of resolvers.stylesheets)
      addStylesheet(path, s);
    effects.output.write(`${faint("in")} ${(elapsed >= 100 ? yellow : faint)(`${elapsed}ms`)}
`);
    pagePaths.add(path);
    outputs.set(path, { type: "page", page, resolvers });
    ++pageCount;
  }
  const outputCount = pageCount + assetCount;
  if (!outputCount)
    throw new CliError(`Nothing to build: no pages found in your ${root} directory.`);
  if (pageCount)
    effects.logger.log(`${faint("built")} ${pageCount} ${faint(`page${pageCount === 1 ? "" : "s"} in`)} ${root}`);
  if (assetCount)
    effects.logger.log(`${faint("built")} ${assetCount} ${faint(`asset${assetCount === 1 ? "" : "s"} in`)} ${root}`);
  const aliases = /* @__PURE__ */ new Map();
  const cacheRoot = join(root, ".observablehq", "cache");
  if (config.search) {
    globalImports.add("/_observablehq/search.js").add("/_observablehq/minisearch.json");
    const contents = await searchIndex(config, pagePaths, effects);
    effects.output.write(`${faint("index \u2192")} `);
    const cachePath = join(cacheRoot, "_observablehq", "minisearch.json");
    await prepareOutput(cachePath);
    await writeFile(cachePath, contents);
    effects.logger.log(cachePath);
  }
  for (const path of globalImports) {
    if (path.startsWith("/_duckdb/")) {
      const sourcePath = join(cacheRoot, path);
      effects.output.write(`${faint("build")} ${path} ${faint("\u2192")} `);
      const contents = await readFile(sourcePath);
      const hash = createHash("sha256").update(contents).digest("hex").slice(0, 8);
      const [, , , version, bundle, name] = path.split("/");
      const alias = join("/_duckdb/", `${basename(name, ".duckdb_extension.wasm")}-${hash}`, version, bundle, name);
      aliases.set(path, alias);
      await effects.writeFile(alias, contents);
    }
  }
  for (const path of globalImports) {
    if (path.startsWith("/_observablehq/") && path.endsWith(".js")) {
      const cachePath = join(cacheRoot, path);
      effects.output.write(`${faint("bundle")} ${path} ${faint("\u2192")} `);
      const clientPath = getClientPath(path === "/_observablehq/client.js" ? "index.js" : path.slice("/_observablehq/".length));
      const define = {};
      if (path === "/_observablehq/stdlib/duckdb.js")
        define["DUCKDB_MANIFEST"] = JSON.stringify(await getDuckDBManifest(duckdb, { root, aliases }));
      const contents = await rollupClient(clientPath, root, path, { minify: true, keepNames: true, define });
      await prepareOutput(cachePath);
      await writeFile(cachePath, contents);
      effects.logger.log(cachePath);
    }
  }
  for (const specifier of stylesheets) {
    if (specifier.startsWith("observablehq:")) {
      let contents;
      const path = `/_observablehq/${specifier.slice("observablehq:".length)}`;
      effects.output.write(`${faint("build")} ${path} ${faint("\u2192")} `);
      if (specifier.startsWith("observablehq:theme-")) {
        const match = /^observablehq:theme-(?<theme>[\w-]+(,[\w-]+)*)?\.css$/.exec(specifier);
        contents = await bundleStyles({ theme: match.groups.theme?.split(",") ?? [], minify: true });
      } else {
        const clientPath = getClientPath(path.slice("/_observablehq/".length));
        contents = await bundleStyles({ path: clientPath, minify: true });
      }
      const hash = createHash("sha256").update(contents).digest("hex").slice(0, 8);
      const alias = applyHash(path, hash);
      aliases.set(path, alias);
      await effects.writeFile(alias, contents);
    } else if (specifier.startsWith("npm:")) {
      effects.output.write(`${faint("copy")} ${specifier} ${faint("\u2192")} `);
      const path = await resolveNpmImport(root, specifier.slice("npm:".length));
      const sourcePath = await populateNpmCache(root, path);
      await effects.copyFile(sourcePath, path);
    } else if (!/^\w+:/.test(specifier)) {
      const sourcePath = join(root, specifier);
      effects.output.write(`${faint("build")} ${sourcePath} ${faint("\u2192")} `);
      const contents = await bundleStyles({ path: sourcePath, minify: true });
      const hash = createHash("sha256").update(contents).digest("hex").slice(0, 8);
      const alias = applyHash(join("/_import", specifier), hash);
      aliases.set(resolveStylesheetPath(root, specifier), alias);
      await effects.writeFile(alias, contents);
    }
  }
  for (const file of files) {
    effects.output.write(`${faint("copy")} ${join(root, file)} ${faint("\u2192")} `);
    const path = join("/", file);
    const loader = loaders.find(path);
    if (!loader)
      throw enoent(path);
    const sourcePath = join(root, await loader.load({ useStale: true }, effects));
    const contents = await readFile(sourcePath);
    const hash = createHash("sha256").update(contents).digest("hex").slice(0, 8);
    const alias = applyHash(join("/_file", file), hash);
    aliases.set(loaders.resolveFilePath(file), alias);
    await effects.writeFile(alias, contents);
  }
  for (const path of globalImports) {
    if (path.endsWith(".js") || path.startsWith("/_duckdb/"))
      continue;
    const sourcePath = join(cacheRoot, path);
    effects.output.write(`${faint("build")} ${path} ${faint("\u2192")} `);
    if (path.startsWith("/_observablehq/")) {
      const contents = await readFile(sourcePath, "utf-8");
      const hash = createHash("sha256").update(contents).digest("hex").slice(0, 8);
      const alias = applyHash(path, hash);
      aliases.set(path, alias);
      await effects.writeFile(alias, contents);
    } else {
      await effects.copyFile(sourcePath, path);
    }
  }
  for (const path of globalImports) {
    if (!path.endsWith(".js"))
      continue;
    const hash = getModuleHash(cacheRoot, path).slice(0, 8);
    const alias = applyHash(path, hash);
    effects.logger.log(`${faint("alias")} ${path} ${faint("\u2192")} ${alias}`);
    aliases.set(path, alias);
  }
  for (const path of globalImports) {
    if (!path.endsWith(".js"))
      continue;
    const sourcePath = join(cacheRoot, path);
    effects.output.write(`${faint("build")} ${path} ${faint("\u2192")} `);
    const resolveImport = (i) => isPathImport(i) ? relativePath(path, aliases.get(i = resolvePath(path, i)) ?? i) : i;
    await effects.writeFile(aliases.get(path), rewriteNpmImports(await readFile(sourcePath, "utf-8"), resolveImport));
  }
  const resolveLocalImport = async (path) => {
    const hash = (await loaders.getLocalModuleHash(path)).slice(0, 8);
    return applyHash(join("/_import", path), hash);
  };
  for (const path of localImports) {
    if (!path.endsWith(".js"))
      continue;
    const module = findModule(root, path);
    if (!module)
      throw new Error(`import not found: ${path}`);
    const sourcePath = join(root, module.path);
    const importPath = join("_import", module.path);
    effects.output.write(`${faint("copy")} ${sourcePath} ${faint("\u2192")} `);
    const resolveImport = loaders.getModuleResolver(path);
    const input = await readJavaScript(sourcePath);
    const contents = await transpileModule(input, {
      root,
      path,
      params: module.params,
      resolveFile(name) {
        const resolution = loaders.resolveFilePath(resolvePath(path, name));
        return aliases.get(resolution) ?? resolution;
      },
      resolveFileInfo(name) {
        return loaders.getOutputInfo(resolvePath(path, name));
      },
      async resolveImport(specifier) {
        let resolution;
        if (isPathImport(specifier)) {
          resolution = await resolveLocalImport(resolvePath(path, specifier));
        } else {
          resolution = await resolveImport(specifier);
          if (isPathImport(resolution)) {
            resolution = resolvePath(importPath, resolution);
            resolution = aliases.get(resolution) ?? resolution;
          }
        }
        return relativePath(importPath, resolution);
      }
    });
    const alias = await resolveLocalImport(path);
    aliases.set(loaders.resolveImportPath(path), alias);
    await effects.writeFile(alias, contents);
  }
  for (const [path, output] of outputs) {
    const { resolvers } = output;
    outputs.set(path, {
      ...output,
      resolvers: {
        ...resolvers,
        resolveFile(specifier) {
          const r = resolvers.resolveFile(specifier);
          const a = aliases.get(resolvePath(path, r));
          return a ? relativePath(path, a) : specifier;
        },
        resolveStylesheet(specifier) {
          const r = resolvers.resolveStylesheet(specifier);
          const a = aliases.get(resolvePath(path, r));
          return a ? relativePath(path, a) : isPathImport(specifier) ? specifier : r;
        },
        resolveImport(specifier) {
          const r = resolvers.resolveImport(specifier);
          const a = aliases.get(resolvePath(path, r));
          return a ? relativePath(path, a) : isPathImport(specifier) ? specifier : r;
        },
        resolveScript(specifier) {
          const r = resolvers.resolveScript(specifier);
          const a = aliases.get(resolvePath(path, r));
          return a ? relativePath(path, a) : specifier;
        }
      }
    });
  }
  for (const [path, output] of outputs) {
    effects.output.write(`${faint("render")} ${path} ${faint("\u2192")} `);
    if (output.type === "page") {
      const { page, resolvers } = output;
      const html = await renderPage(page, { ...config, path, resolvers });
      await effects.writeFile(`${path}.html`, html);
      addToManifest("pages", path, page);
    } else {
      const { resolvers } = output;
      const source = await renderModule(root, path, resolvers);
      await effects.writeFile(path, source);
    }
  }
  await effects.writeBuildManifest(buildManifest);
  const columnWidth = 12;
  effects.logger.log("");
  for (const [indent, name, description, node] of tree(outputs)) {
    if (node.children) {
      effects.logger.log(
        `${faint(indent)}${name}${faint(description)} ${node.depth ? "" : ["Page", "Imports", "Files"].map((name2) => name2.padStart(columnWidth)).join(" ")}`
      );
    } else {
      const [path, { type, resolvers }] = node.data;
      const resolveOutput = (name2) => join(config.output, resolvePath(path, name2));
      const pageSize = (await stat(join(config.output, type === "page" ? `${path}.html` : path))).size;
      const importSize = await accumulateSize(resolvers.staticImports, resolvers.resolveImport, resolveOutput);
      const fileSize = await accumulateSize(resolvers.files, resolvers.resolveFile, resolveOutput) + await accumulateSize(resolvers.assets, resolvers.resolveFile, resolveOutput) + await accumulateSize(resolvers.stylesheets, resolvers.resolveStylesheet, resolveOutput);
      effects.logger.log(
        `${faint(indent)}${name}${description} ${[pageSize, importSize, fileSize].map((size) => formatBytes(size, columnWidth)).join(" ")}`
      );
    }
  }
  effects.logger.log("");
  const [validLinks, brokenLinks] = validateLinks(outputs);
  if (brokenLinks.length) {
    effects.logger.warn(`${yellow("Warning: ")}${brokenLinks.length} broken link${brokenLinks.length === 1 ? "" : "s"} (${validLinks.length + brokenLinks.length} validated)`);
    for (const [path, link] of brokenLinks)
      effects.logger.log(`${faint("\u21B3")} ${path} ${faint("\u2192")} ${red(link)}`);
  } else if (validLinks.length) {
    effects.logger.log(`${green(`${validLinks.length}`)} link${validLinks.length === 1 ? "" : "s"} validated`);
  }
  Telemetry.record({ event: "build", step: "finish", pageCount });
}
function validateLinks(outputs) {
  const validTargets = new Set(outputs.keys());
  for (const [path, { resolvers }] of outputs) {
    for (const anchor of resolvers.anchors) {
      validTargets.add(`${path}#${encodeURIComponent(anchor)}`);
    }
  }
  const valid = [];
  const broken = [];
  for (const [path, { resolvers }] of outputs) {
    for (const target of resolvers.localLinks) {
      (validTargets.has(target) ? valid : broken).push([path, target]);
    }
  }
  return [valid, broken];
}
function applyHash(path, hash) {
  const ext = extname(path);
  let name = basename(path, ext);
  if (path.endsWith(".js"))
    name = name.replace(/(^|\.)_esm$/, "");
  return join(dirname(path), `${name && `${name}.`}${hash}${ext}`);
}
async function accumulateSize(files, resolveFile, resolveOutput) {
  let size = 0;
  for (const file of files) {
    const fileResolution = resolveFile(file);
    if (isAssetPath(fileResolution)) {
      try {
        size += (await stat(resolveOutput(fileResolution))).size;
      } catch {
      }
    }
  }
  return size;
}
function formatBytes(size, length, locale = "en-US") {
  let color;
  let text;
  if (size < 1e3) {
    text = "<1 kB";
    color = faint;
  } else if (size < 1e6) {
    text = (size / 1e3).toLocaleString(locale, { maximumFractionDigits: 0 }) + " kB";
    color = green;
  } else {
    text = (size / 1e6).toLocaleString(locale, { minimumFractionDigits: 3, maximumFractionDigits: 3 }) + " MB";
    color = size < 1e7 ? yellow : size < 5e7 ? magenta : red;
  }
  return color(text.padStart(length));
}
class FileBuildEffects {
  outputRoot;
  cacheDir;
  logger;
  output;
  constructor(outputRoot, cacheDir, { logger = console, output = process.stdout } = {}) {
    if (!outputRoot)
      throw new Error("missing outputRoot");
    this.logger = logger;
    this.output = output;
    this.outputRoot = outputRoot;
    this.cacheDir = cacheDir;
  }
  async prepare() {
    if (within(process.cwd(), this.outputRoot)) {
      await rm(this.outputRoot, { recursive: true, force: true });
    } else {
      this.logger.warn(
        `${yellow("Warning:")} the output root ${this.outputRoot} is not within the current working directory and will not be cleared.`
      );
    }
  }
  async copyFile(sourcePath, outputPath) {
    const destination = join(this.outputRoot, outputPath);
    this.logger.log(destination);
    await prepareOutput(destination);
    if (existsSync(destination))
      throw new Error(`file conflict: ${outputPath}`);
    await copyFile(sourcePath, destination);
  }
  async writeFile(outputPath, contents) {
    const destination = join(this.outputRoot, outputPath);
    this.logger.log(destination);
    await prepareOutput(destination);
    if (existsSync(destination))
      throw new Error(`file conflict: ${outputPath}`);
    await writeFile(destination, contents);
  }
  async writeBuildManifest(buildManifest) {
    const destination = join(this.cacheDir, "_build.json");
    await prepareOutput(destination);
    await writeFile(destination, JSON.stringify(buildManifest));
  }
}
export {
  FileBuildEffects,
  build
};
