import { createHash } from "node:crypto";
import { existsSync, readFileSync } from "node:fs";
import { stat } from "node:fs/promises";
import op from "node:path";
import { basename, dirname, extname, join } from "node:path/posix";
import { cwd } from "node:process";
import { pathToFileURL } from "node:url";
import he from "he";
import wrapAnsi from "wrap-ansi";
import { DUCKDB_CORE_ALIASES, DUCKDB_CORE_EXTENSIONS } from "./duckdb.js";
import { visitFiles } from "./files.js";
import { formatIsoDate, formatLocaleDate } from "./format.js";
import { findModule } from "./javascript/module.js";
import { LoaderResolver } from "./loader.js";
import { createMarkdownIt, parseMarkdownMetadata } from "./markdown.js";
import { getPagePaths } from "./pager.js";
import { isAssetPath, parseRelativeUrl, resolvePath } from "./path.js";
import { isParameterized } from "./route.js";
import { resolveTheme } from "./theme.js";
import { bold, yellow } from "./tty.js";
function resolveConfig(configPath, root = ".") {
  return op.join(cwd(), root, configPath);
}
async function importConfig(path) {
  const { mtimeMs } = await stat(path);
  return (await import(`${pathToFileURL(path).href}?${mtimeMs}`)).default;
}
async function readConfig(configPath, root) {
  if (configPath === void 0)
    configPath = await resolveDefaultConfig(root);
  if (configPath === void 0)
    return normalizeConfig(void 0, root);
  return normalizeConfig(await importConfig(configPath), root, configPath);
}
async function resolveDefaultConfig(root) {
  const jsPath = resolveConfig("observablehq.config.js", root);
  if (existsSync(jsPath))
    return jsPath;
  const tsPath = resolveConfig("observablehq.config.ts", root);
  if (existsSync(tsPath))
    return await import("tsx/esm"), tsPath;
}
let cachedPages = null;
function readPages(root, md) {
  const files = [];
  const hash = createHash("sha256");
  for (const file of visitFiles(root, (name) => !isParameterized(name))) {
    if (extname(file) !== ".md" || file === "index.md" || file === "404.md")
      continue;
    const path = file.slice(0, -".md".length);
    if (path.endsWith(".js") && findModule(root, path))
      continue;
    const source = readFileSync(join(root, file), "utf8");
    files.push({ file, source });
    hash.update(file).update(source);
  }
  const key = hash.digest("hex");
  if (cachedPages?.key === key)
    return cachedPages.pages;
  const pages = [];
  for (const { file, source } of files) {
    const { data, title } = parseMarkdownMetadata(source, { path: file, md });
    if (data.draft)
      continue;
    const name = basename(file, ".md");
    const { pager = "main" } = data;
    const page = { path: join("/", dirname(file), name), name: title ?? "Untitled", pager };
    if (name === "index")
      pages.unshift(page);
    else
      pages.push(page);
  }
  cachedPages = { key, pages };
  return pages;
}
let currentDate = null;
function setCurrentDate(date) {
  currentDate = date;
}
const configCache = /* @__PURE__ */ new WeakMap();
function normalizeConfig(spec = {}, defaultRoot, watchPath) {
  const cachedConfig = configCache.get(spec);
  if (cachedConfig)
    return cachedConfig;
  const root = spec.root === void 0 ? findDefaultRoot(defaultRoot) : String(spec.root);
  const output = spec.output === void 0 ? "dist" : String(spec.output);
  const base = spec.base === void 0 ? "/" : normalizeBase(spec.base);
  const style = spec.style === null ? null : spec.style !== void 0 ? { path: String(spec.style) } : { theme: normalizeTheme(spec.theme === void 0 ? "default" : spec.theme) };
  const globalStylesheets = spec.globalStylesheets === void 0 ? defaultGlobalStylesheets() : normalizeGlobalStylesheets(spec.globalStylesheets);
  const md = createMarkdownIt({
    linkify: spec.linkify === void 0 ? void 0 : Boolean(spec.linkify),
    typographer: spec.typographer === void 0 ? void 0 : Boolean(spec.typographer),
    quotes: spec.quotes === void 0 ? void 0 : spec.quotes,
    markdownIt: spec.markdownIt
  });
  const title = spec.title === void 0 ? void 0 : String(spec.title);
  const home = spec.home === void 0 ? he.escape(title ?? "Home") : String(spec.home);
  const pages = spec.pages === void 0 ? void 0 : normalizePages(spec.pages);
  const pager = spec.pager === void 0 ? true : Boolean(spec.pager);
  const dynamicPaths = normalizeDynamicPaths(spec.dynamicPaths);
  const toc = normalizeToc(spec.toc);
  const sidebar = spec.sidebar === void 0 ? void 0 : Boolean(spec.sidebar);
  const scripts = spec.scripts === void 0 ? [] : normalizeScripts(spec.scripts);
  const head = pageFragment(spec.head === void 0 ? "" : spec.head);
  const header = pageFragment(spec.header === void 0 ? "" : spec.header);
  const footer = pageFragment(spec.footer === void 0 ? defaultFooter() : spec.footer);
  const search = spec.search == null || spec.search === false ? null : normalizeSearch(spec.search);
  const interpreters = normalizeInterpreters(spec.interpreters);
  const normalizePath2 = getPathNormalizer(spec);
  const duckdb = normalizeDuckDB(spec.duckdb);
  function normalizePagePath(pathname) {
    ({ pathname } = parseRelativeUrl(pathname));
    pathname = normalizePath2(pathname);
    if (pathname.endsWith("/"))
      pathname = join(pathname, "index");
    else
      pathname = pathname.replace(/\.html$/, "");
    return pathname;
  }
  const config = {
    root,
    output,
    base,
    home,
    title,
    sidebar,
    // see below
    pages,
    // see below
    pager,
    async *paths() {
      const visited = /* @__PURE__ */ new Set();
      function* visit(path) {
        if (!visited.has(path = normalizePagePath(path))) {
          visited.add(path);
          yield path;
        }
      }
      for (const path of this.loaders.findPagePaths()) {
        yield* visit(path);
      }
      for (const path of getPagePaths(this)) {
        yield* visit(path);
      }
      for await (const path of dynamicPaths()) {
        yield* visit(path);
      }
    },
    scripts,
    head,
    header,
    footer,
    toc,
    style,
    globalStylesheets,
    search,
    md,
    normalizePath: normalizePath2,
    loaders: new LoaderResolver({ root, interpreters }),
    watchPath,
    duckdb
  };
  if (pages === void 0)
    Object.defineProperty(config, "pages", { get: () => readPages(root, md) });
  if (sidebar === void 0)
    Object.defineProperty(config, "sidebar", { get: () => config.pages.length > 0 });
  configCache.set(spec, config);
  return config;
}
function normalizeDynamicPaths(spec) {
  if (typeof spec === "function")
    return spec;
  const paths = Array.from(spec ?? [], String);
  return async function* () {
    yield* paths;
  };
}
function normalizeCleanUrls(spec) {
  console.warn(`${yellow("Warning:")} the ${bold("cleanUrls")} option is deprecated; use ${bold("preserveIndex")} and ${bold("preserveExtension")} instead.`);
  return !spec;
}
function getPathNormalizer(spec) {
  const preserveIndex = spec.preserveIndex !== void 0 ? Boolean(spec.preserveIndex) : false;
  const preserveExtension = spec.preserveExtension !== void 0 ? Boolean(spec.preserveExtension) : spec.cleanUrls !== void 0 ? normalizeCleanUrls(spec.cleanUrls) : false;
  return (path) => {
    const ext = extname(path);
    if (path.endsWith("."))
      path += "/";
    if (ext === ".html")
      path = path.slice(0, -".html".length);
    if (path.endsWith("/index"))
      path = path.slice(0, -"index".length);
    if (preserveIndex && path.endsWith("/"))
      path += "index";
    if (!preserveIndex && path === "index")
      path = ".";
    if (preserveExtension && path && !path.endsWith(".") && !path.endsWith("/") && !extname(path))
      path += ".html";
    return path;
  };
}
function pageFragment(spec) {
  return typeof spec === "function" ? spec : stringOrNull(spec);
}
function defaultGlobalStylesheets() {
  return [
    "https://fonts.googleapis.com/css2?family=Source+Serif+4:ital,opsz,wght@0,8..60,200..900;1,8..60,200..900&display=swap"
  ];
}
function defaultFooter() {
  const date = currentDate ?? /* @__PURE__ */ new Date();
  return `Built with <a href="https://observablehq.com/" target="_blank">Observable</a> on <a title="${formatIsoDate(
    date
  )}">${formatLocaleDate(date)}</a>.`;
}
function findDefaultRoot(defaultRoot) {
  if (defaultRoot !== void 0)
    return defaultRoot;
  const root = existsSync("docs") ? "docs" : "src";
  console.warn(
    wrapAnsi(
      `${yellow("Warning:")} the config file is missing the ${bold(
        "root"
      )} option, which specifies the path to the source root.${root === "docs" ? ` The recommended source root is ${bold('"src"')}; however, since ${bold(
        "docs"
      )} exists and was previously the default for this option, we will use ${bold('"docs"')}.` : ""} You can suppress this warning by specifying ${bold(`root: ${JSON.stringify(root)}`)} in the config file.
`,
      Math.min(80, process.stdout.columns ?? 80)
    )
  );
  return root;
}
function normalizeArray(spec, f) {
  return spec == null ? [] : Array.from(spec, f);
}
function normalizeBase(spec) {
  let base = String(spec);
  if (!base.startsWith("/"))
    throw new Error(`base must start with slash: ${base}`);
  if (!base.endsWith("/"))
    base += "/";
  return base;
}
function normalizeGlobalStylesheets(spec) {
  return normalizeArray(spec, String);
}
function normalizeTheme(spec) {
  return resolveTheme(typeof spec === "string" ? [spec] : normalizeArray(spec, String));
}
function normalizeScripts(spec) {
  console.warn(`${yellow("Warning:")} the ${bold("scripts")} option is deprecated; use ${bold("head")} instead.`);
  return normalizeArray(spec, normalizeScript);
}
function normalizeScript(spec) {
  const script = typeof spec === "string" ? { src: spec } : spec;
  const src = String(script.src);
  const async = script.async === void 0 ? false : Boolean(script.async);
  const type = script.type == null ? null : String(script.type);
  return { src, async, type };
}
function normalizePages(spec) {
  return normalizeArray(
    spec,
    (spec2) => "pages" in spec2 ? normalizeSection(spec2, normalizePage) : normalizePage(spec2)
  );
}
function normalizeSection(spec, normalizePage2) {
  const name = String(spec.name);
  const collapsible = spec.collapsible === void 0 ? spec.open !== void 0 : Boolean(spec.collapsible);
  const open = collapsible ? Boolean(spec.open) : true;
  const pager = spec.pager === void 0 ? "main" : stringOrNull(spec.pager);
  const path = spec.path == null ? null : normalizePath(spec.path);
  const pages = normalizeArray(spec.pages, (spec2) => normalizePage2(spec2, pager));
  return { name, collapsible, open, path, pager, pages };
}
function normalizePage(spec, defaultPager = "main") {
  const name = String(spec.name);
  const path = normalizePath(spec.path);
  const pager = spec.pager === void 0 && isAssetPath(path) ? defaultPager : stringOrNull(spec.pager);
  return { name, path, pager };
}
function normalizeSearch(spec) {
  const index = spec.index == null ? null : spec.index;
  if (index !== null && typeof index !== "function")
    throw new Error("search.index is not a function");
  return { index };
}
function normalizePath(spec) {
  let path = String(spec);
  if (isAssetPath(path)) {
    const u = parseRelativeUrl(join("/", path));
    let { pathname } = u;
    pathname = pathname.replace(/\.html$/i, "");
    pathname = pathname.replace(/\/$/, "/index");
    path = pathname + u.search + u.hash;
  }
  return path;
}
function normalizeInterpreters(spec = {}) {
  return Object.fromEntries(
    Object.entries(spec).map(([key, value]) => {
      return [String(key), normalizeArray(value, String)];
    })
  );
}
function normalizeToc(spec = true) {
  const toc = typeof spec === "boolean" ? { show: spec } : spec;
  const label = toc.label === void 0 ? "Contents" : String(toc.label);
  const show = toc.show === void 0 ? true : Boolean(toc.show);
  return { label, show };
}
function mergeToc(spec = {}, toc) {
  const { label = toc.label, show = toc.show } = spec;
  return { label, show };
}
function mergeStyle(path, style, theme, defaultStyle) {
  return style === void 0 && theme === void 0 ? defaultStyle : style === null ? null : style !== void 0 ? { path: resolvePath(path, style) } : theme === void 0 ? defaultStyle : { theme };
}
function stringOrNull(spec) {
  return spec == null || spec === false ? null : String(spec);
}
function normalizeDuckDB(spec) {
  const { mvp = true, eh = true } = spec?.["platforms"] ?? {};
  const extensions = {};
  let extspec = spec?.["extensions"] ?? {};
  if (Array.isArray(extspec))
    extspec = Object.fromEntries(extspec.map((name) => [name, {}]));
  if (extspec.json === void 0)
    extspec = { ...extspec, json: false };
  if (extspec.parquet === void 0)
    extspec = { ...extspec, parquet: false };
  for (let name in extspec) {
    if (!/^\w+$/.test(name))
      throw new Error(`invalid extension: ${name}`);
    const vspec = extspec[name];
    if (vspec == null)
      continue;
    name = DUCKDB_CORE_ALIASES[name] ?? name;
    const {
      source = name in DUCKDB_CORE_EXTENSIONS ? "core" : "community",
      install = true,
      load = !DUCKDB_CORE_EXTENSIONS[name]
    } = typeof vspec === "boolean" ? { load: vspec } : typeof vspec === "string" ? { source: vspec } : vspec;
    extensions[name] = {
      source: normalizeDuckDBSource(String(source)),
      install: Boolean(install),
      load: Boolean(load)
    };
  }
  return {
    platforms: Object.fromEntries(
      [
        ["mvp", mvp],
        ["eh", eh]
      ].filter(([, enabled]) => enabled)
    ),
    extensions
  };
}
function normalizeDuckDBSource(source) {
  if (source === "core")
    return "https://extensions.duckdb.org/";
  if (source === "community")
    return "https://community-extensions.duckdb.org/";
  const url = new URL(source);
  if (url.protocol !== "https:")
    throw new Error(`invalid source: ${source}`);
  return String(url);
}
export {
  currentDate,
  mergeStyle,
  mergeToc,
  normalizeConfig,
  normalizeTheme,
  readConfig,
  setCurrentDate,
  stringOrNull
};
