import { existsSync } from "node:fs";
import { mkdir, writeFile } from "node:fs/promises";
import { dirname, join } from "node:path/posix";
import { faint } from "./tty.js";
const downloadRequests = /* @__PURE__ */ new Map();
const DUCKDB_WASM_VERSION = "1.29.0";
const DUCKDB_VERSION = "1.1.1";
const DUCKDB_CORE_ALIASES = {
  sqlite: "sqlite_scanner",
  sqlite3: "sqlite_scanner",
  postgres_scanner: "postgres",
  http: "httpfs",
  https: "httpfs",
  s3: "httpfs"
};
const DUCKDB_CORE_EXTENSIONS = {
  arrow: false,
  autocomplete: true,
  aws: true,
  azure: true,
  delta: true,
  excel: true,
  fts: true,
  httpfs: true,
  iceberg: false,
  icu: true,
  inet: true,
  jemalloc: false,
  json: true,
  mysql: false,
  parquet: true,
  postgres: true,
  spatial: false,
  sqlite_scanner: true,
  substrait: false,
  tpcds: true,
  tpch: true,
  vss: false
};
async function getDuckDBManifest({ platforms, extensions }, { root, aliases }) {
  return {
    platforms: { mvp: "mvp" in platforms, eh: "eh" in platforms },
    extensions: Object.fromEntries(
      await Promise.all(
        Object.entries(extensions).map(
          ([name, { install, load, source }]) => (async () => [
            name,
            {
              install,
              load,
              ...Object.fromEntries(
                await Promise.all(
                  Object.keys(platforms).map(async (platform) => [
                    platform,
                    install ? await getDuckDBExtension(root, resolveDuckDBExtension(source, platform, name), aliases) : source
                  ])
                )
              )
            }
          ])()
        )
      )
    )
  };
}
function resolveDuckDBExtension(repo, platform, name) {
  return new URL(`v${DUCKDB_VERSION}/wasm_${platform}/${name}.duckdb_extension.wasm`, repo);
}
async function getDuckDBExtension(root, href, aliases) {
  let ext = await cacheDuckDBExtension(root, href);
  if (aliases?.has(ext))
    ext = aliases.get(ext);
  return join("..", "..", dirname(dirname(dirname(ext))));
}
async function cacheDuckDBExtension(root, href) {
  const url = new URL(href);
  if (url.protocol !== "https:")
    throw new Error(`unsupported protocol: ${url.protocol}`);
  const key = String(url).slice("https://".length);
  const path = join("_duckdb", key);
  const cache = join(root, ".observablehq", "cache");
  const cachePath = join(cache, path);
  if (existsSync(cachePath))
    return `/${path}`;
  let promise = downloadRequests.get(cachePath);
  if (promise)
    return promise;
  promise = (async () => {
    console.log(`duckdb:${key} ${faint("\u2192")} ${cachePath}`);
    const response = await fetch(url);
    if (!response.ok)
      throw new Error(`unable to fetch: ${url}`);
    await mkdir(dirname(cachePath), { recursive: true });
    await writeFile(cachePath, Buffer.from(await response.arrayBuffer()));
    return `/${path}`;
  })();
  promise.catch(console.error).then(() => downloadRequests.delete(cachePath));
  downloadRequests.set(cachePath, promise);
  return promise;
}
export {
  DUCKDB_CORE_ALIASES,
  DUCKDB_CORE_EXTENSIONS,
  DUCKDB_VERSION,
  DUCKDB_WASM_VERSION,
  cacheDuckDBExtension,
  getDuckDBManifest,
  resolveDuckDBExtension
};
