import assert from "node:assert";
class HttpError extends Error {
  statusCode;
  details;
  constructor(message, statusCode, options) {
    const { details, ...errorOptions } = options ?? {};
    super(message, errorOptions);
    this.statusCode = statusCode;
    this.details = details;
    Error.captureStackTrace(this, HttpError);
  }
}
function enoent(path) {
  return Object.assign(new Error(`Not found: ${path}`), { code: "ENOENT" });
}
function isEnoent(error) {
  return isSystemError(error) && error.code === "ENOENT";
}
function isSystemError(error) {
  return error instanceof Error && "code" in error;
}
function isHttpError(error) {
  return error instanceof Error && "statusCode" in error;
}
function isApiError(error) {
  return isHttpError(error) && !!error.details && typeof error.details === "object" && "errors" in error.details && Array.isArray(error.details.errors) && error.details.errors.every((e) => typeof e === "object" && "code" in e);
}
class CliError extends Error {
  exitCode;
  print;
  constructor(message, { exitCode = 1, print = true, ...options } = {}) {
    super(message, options);
    this.exitCode = exitCode;
    this.print = print;
  }
  /** Use in tests to check if a thrown error is the error you expected. */
  static assert(error, { message, exitCode = 1, print = true } = {}) {
    assert.ok(error instanceof Error, `Expected error to be an Error but got ${error}`);
    assert.ok(error instanceof CliError, `Expected error to be a CliError but got ${error.message}
${error.stack}`);
    if (typeof message === "string") {
      assert.equal(error.message, message);
    } else if (message instanceof RegExp) {
      assert.ok(
        message.test(error.message),
        `Expected error message to match regexp ${message.toString()} but got ${error.message}`
      );
    }
    assert.equal(error.exitCode, exitCode, `Expected exit code to be ${exitCode}, but got ${error.exitCode}`);
    assert.equal(error.print, print, `Expected print to be ${print}, but got ${error.print}`);
  }
  /** Use in tests to check if a thrown error is the error you expected. */
  static match(error, { message, exitCode, print } = {}) {
    if (!(error instanceof Error))
      return false;
    if (!(error instanceof CliError))
      return false;
    if (message !== void 0 && typeof message === "string" && error.message !== message)
      return false;
    if (message !== void 0 && message instanceof RegExp && !message.test(error.message))
      return false;
    if (exitCode !== void 0 && error.exitCode !== exitCode)
      return false;
    if (print !== void 0 && error.print !== print)
      return false;
    return true;
  }
}
export {
  CliError,
  HttpError,
  enoent,
  isApiError,
  isEnoent,
  isHttpError,
  isSystemError
};
