import matter from "gray-matter";
import { normalizeTheme, stringOrNull } from "./config.js";
import { yellow } from "./tty.js";
function readFrontMatter(input) {
  try {
    const { content, data } = matter(input, {});
    return { content, data: normalizeFrontMatter(data) };
  } catch (error) {
    if ("mark" in error) {
      console.warn(`${yellow("Invalid front matter:")} ${error.reason}`);
      return { data: {}, content: input };
    }
    throw error;
  }
}
function normalizeFrontMatter(spec = {}) {
  const frontMatter = {};
  if (spec == null || typeof spec !== "object")
    return frontMatter;
  const { title, sidebar, toc, index, keywords, draft, sql, head, header, footer, pager, style, theme } = spec;
  if (title !== void 0)
    frontMatter.title = stringOrNull(title);
  if (sidebar !== void 0)
    frontMatter.sidebar = Boolean(sidebar);
  if (toc !== void 0)
    frontMatter.toc = normalizeToc(toc);
  if (index !== void 0)
    frontMatter.index = Boolean(index);
  if (keywords !== void 0)
    frontMatter.keywords = normalizeKeywords(keywords);
  if (draft !== void 0)
    frontMatter.draft = Boolean(draft);
  if (sql !== void 0)
    frontMatter.sql = normalizeSql(sql);
  if (head !== void 0)
    frontMatter.head = stringOrNull(head);
  if (header !== void 0)
    frontMatter.header = stringOrNull(header);
  if (footer !== void 0)
    frontMatter.footer = stringOrNull(footer);
  if (pager !== void 0)
    frontMatter.pager = stringOrNull(pager);
  if (style !== void 0)
    frontMatter.style = stringOrNull(style);
  if (theme !== void 0)
    frontMatter.theme = normalizeTheme(theme);
  return frontMatter;
}
function normalizeToc(spec) {
  if (spec == null)
    return { show: false };
  if (typeof spec !== "object")
    return { show: Boolean(spec) };
  const { show, label } = spec;
  const toc = {};
  if (show !== void 0)
    toc.show = Boolean(show);
  if (label !== void 0)
    toc.label = String(label);
  return toc;
}
function normalizeKeywords(spec) {
  return spec == null ? [] : typeof spec === "string" ? [spec] : Array.from(spec, String);
}
function normalizeSql(spec) {
  if (spec == null || typeof spec !== "object")
    return {};
  const sql = {};
  for (const key in spec)
    sql[key] = String(spec[key]);
  return sql;
}
export {
  normalizeFrontMatter,
  readFrontMatter
};
