import { extname } from "node:path/posix";
import { ancestor, simple } from "acorn-walk";
import { relativePath, resolveLocalPath } from "../path.js";
import { defaultGlobals } from "./globals.js";
import { findReferences } from "./references.js";
import { getStringLiteralValue, isStringLiteral } from "./source.js";
import { syntaxError } from "./syntaxError.js";
const KNOWN_FILE_EXTENSIONS = {
  ".arrow": "arrow",
  ".csv": "csv",
  ".db": "sqlite",
  ".html": "html",
  ".json": "json",
  ".parquet": "parquet",
  ".sqlite": "sqlite",
  ".tsv": "tsv",
  ".txt": "text",
  ".xlsx": "xlsx",
  ".xml": "xml",
  ".zip": "zip"
};
function findFiles(body, path, input, aliases) {
  const declarations = /* @__PURE__ */ new Set();
  const alias = new Set(aliases);
  let globals;
  simple(body, {
    ImportDeclaration(node) {
      if (node.source.value === "observablehq:stdlib" || node.source.value === "npm:@observablehq/stdlib") {
        for (const specifier of node.specifiers) {
          if (specifier.type === "ImportSpecifier" && specifier.imported.type === "Identifier" && specifier.imported.name === "FileAttachment") {
            declarations.add(specifier.local);
            alias.add(specifier.local.name);
          }
        }
      }
    }
  });
  for (const name of alias.keys()) {
    if (defaultGlobals.has(name)) {
      (globals ??= new Set(defaultGlobals)).delete(name);
    }
  }
  const references = new Set(
    findReferences(body, {
      globals,
      filterDeclaration: (identifier) => !declarations.has(identifier)
      // treat the imported declaration as unbound
    })
  );
  const files = [];
  ancestor(body, {
    CallExpression(node, state, stack) {
      const { callee } = node;
      if (callee.type !== "Identifier" || !alias.has(callee.name) || !references.has(callee))
        return;
      const args = node.arguments;
      if (args.length !== 1)
        throw syntaxError("FileAttachment requires a single literal string argument", node, input);
      const [arg] = args;
      if (!isStringLiteral(arg))
        throw syntaxError("FileAttachment requires a single literal string argument", node, input);
      const fileName = getStringLiteralValue(arg);
      const filePath = resolveLocalPath(path, fileName);
      if (!filePath)
        throw syntaxError(`non-local file path: ${fileName}`, node, input);
      const parent = stack[stack.length - 2];
      const name = relativePath(path, filePath);
      const method = parent && isMemberExpression(parent) && parent.property.type === "Identifier" ? parent.property.name === "arquero" && /\.parquet$/i.test(fileName) ? "arquero-parquet" : parent.property.name : KNOWN_FILE_EXTENSIONS[extname(fileName).toLowerCase()];
      files.push({ node, name, method });
    }
  });
  return files;
}
function isMemberExpression(node) {
  return node.type === "MemberExpression";
}
export {
  findFiles
};
