import { readFile } from "node:fs/promises";
import { join } from "node:path/posix";
import { simple } from "acorn-walk";
import { isPathImport, relativePath, resolveLocalPath } from "../path.js";
import { parseProgram } from "./parse.js";
import { getStringLiteralValue, isStringLiteral } from "./source.js";
import { syntaxError } from "./syntaxError.js";
function findExports(body) {
  const exports = [];
  simple(body, {
    ExportAllDeclaration: findExport,
    ExportNamedDeclaration: findExport
  });
  function findExport(node) {
    exports.push(node);
  }
  return exports;
}
function hasImportDeclaration(body) {
  let has = false;
  simple(body, {
    ImportDeclaration() {
      has = true;
    }
  });
  return has;
}
function findImports(body, path, input) {
  const imports = [];
  const keys = /* @__PURE__ */ new Set();
  simple(body, {
    ImportDeclaration: findImport,
    ImportExpression: findImport,
    ExportAllDeclaration: findImport,
    ExportNamedDeclaration: findImport,
    CallExpression: findImportMetaResolve
  });
  function addImport(ref) {
    const key = `${ref.type}:${ref.method}:${ref.name}`;
    if (!keys.has(key))
      keys.add(key), imports.push(ref);
  }
  function findImport(node) {
    const source = node.source;
    if (!source || !isStringLiteral(source))
      return;
    const name = decodeURI(getStringLiteralValue(source));
    const method = node.type === "ImportExpression" ? "dynamic" : "static";
    if (isPathImport(name)) {
      const localPath = resolveLocalPath(path, name);
      if (!localPath)
        throw syntaxError(`non-local import: ${name}`, node, input);
      addImport({ name: relativePath(path, localPath), type: "local", method });
    } else {
      addImport({ name, type: "global", method });
    }
  }
  function findImportMetaResolve(node) {
    const source = node.arguments[0];
    if (!isImportMetaResolve(node) || !isStringLiteral(source))
      return;
    const name = decodeURI(getStringLiteralValue(source));
    if (isPathImport(name)) {
      const localPath = resolveLocalPath(path, name);
      if (!localPath)
        throw syntaxError(`non-local import: ${name}`, node, input);
      addImport({ name: relativePath(path, localPath), type: "local", method: "resolve" });
    } else {
      addImport({ name, type: "global", method: "resolve" });
    }
  }
  return imports;
}
function isImportMetaResolve(node) {
  return node.callee.type === "MemberExpression" && node.callee.object.type === "MetaProperty" && node.callee.object.meta.name === "import" && node.callee.object.property.name === "meta" && node.callee.property.type === "Identifier" && node.callee.property.name === "resolve" && node.arguments.length > 0;
}
function isJavaScript(path) {
  return /\.(m|c)?js(\?|$)/i.test(path);
}
const parseImportsCache = /* @__PURE__ */ new Map();
async function parseImports(root, path) {
  if (!isJavaScript(path))
    return [];
  const filePath = join(root, path);
  let promise = parseImportsCache.get(filePath);
  if (promise)
    return promise;
  promise = async function() {
    try {
      const source = await readFile(filePath, "utf-8");
      const body = parseProgram(source);
      return findImports(body, path, source);
    } catch (error) {
      console.warn(`unable to fetch or parse ${path}: ${error.message}`);
      return [];
    }
  }();
  parseImportsCache.set(filePath, promise);
  return promise;
}
export {
  findExports,
  findImports,
  hasImportDeclaration,
  isImportMetaResolve,
  isJavaScript,
  parseImports
};
