import { ancestor } from "acorn-walk";
import { defaultGlobals } from "./globals.js";
function isScope(node) {
  return node.type === "FunctionExpression" || node.type === "FunctionDeclaration" || node.type === "ArrowFunctionExpression" || node.type === "Program";
}
function isBlockScope(node) {
  return node.type === "BlockStatement" || node.type === "SwitchStatement" || node.type === "ForInStatement" || node.type === "ForOfStatement" || node.type === "ForStatement" || isScope(node);
}
function findReferences(node, {
  globals = defaultGlobals,
  filterReference = (identifier) => !globals.has(identifier.name),
  filterDeclaration = () => true
} = {}) {
  const locals = /* @__PURE__ */ new Map();
  const references = [];
  function hasLocal(node2, name) {
    const l = locals.get(node2);
    return l ? l.has(name) : false;
  }
  function declareLocal(node2, id) {
    if (!filterDeclaration(id))
      return;
    const l = locals.get(node2);
    if (l)
      l.add(id.name);
    else
      locals.set(node2, /* @__PURE__ */ new Set([id.name]));
  }
  function declareClass(node2) {
    if (node2.id)
      declareLocal(node2, node2.id);
  }
  function declareFunction(node2) {
    node2.params.forEach((param) => declarePattern(param, node2));
    if (node2.id)
      declareLocal(node2, node2.id);
    if (node2.type !== "ArrowFunctionExpression")
      declareLocal(node2, { name: "arguments" });
  }
  function declareCatchClause(node2) {
    if (node2.param)
      declarePattern(node2.param, node2);
  }
  function declarePattern(node2, parent) {
    switch (node2.type) {
      case "Identifier":
        declareLocal(parent, node2);
        break;
      case "ObjectPattern":
        node2.properties.forEach((node3) => declarePattern(node3.type === "Property" ? node3.value : node3, parent));
        break;
      case "ArrayPattern":
        node2.elements.forEach((node3) => node3 && declarePattern(node3, parent));
        break;
      case "RestElement":
        declarePattern(node2.argument, parent);
        break;
      case "AssignmentPattern":
        declarePattern(node2.left, parent);
        break;
    }
  }
  ancestor(node, {
    VariableDeclaration(node2, state, parents) {
      let parent = null;
      for (let i = parents.length - 1; i >= 0 && parent === null; --i) {
        if (node2.kind === "var" ? isScope(parents[i]) : isBlockScope(parents[i])) {
          parent = parents[i];
        }
      }
      node2.declarations.forEach((declaration) => declarePattern(declaration.id, parent));
    },
    FunctionDeclaration(node2, state, parents) {
      let parent = null;
      for (let i = parents.length - 2; i >= 0 && parent === null; --i) {
        if (isScope(parents[i])) {
          parent = parents[i];
        }
      }
      if (node2.id)
        declareLocal(parent, node2.id);
      declareFunction(node2);
    },
    FunctionExpression: declareFunction,
    ArrowFunctionExpression: declareFunction,
    ClassDeclaration(node2, state, parents) {
      let parent = null;
      for (let i = parents.length - 2; i >= 0 && parent === null; --i) {
        if (isScope(parents[i])) {
          parent = parents[i];
        }
      }
      if (node2.id)
        declareLocal(parent, node2.id);
    },
    ClassExpression: declareClass,
    CatchClause: declareCatchClause,
    ImportDeclaration(node2, state, [root]) {
      node2.specifiers.forEach((specifier) => declareLocal(root, specifier.local));
    }
  });
  function identifier(node2, state, parents) {
    const name = node2.name;
    if (name === "undefined")
      return;
    for (let i = parents.length - 2; i >= 0; --i) {
      if (hasLocal(parents[i], name)) {
        return;
      }
    }
    if (filterReference(node2)) {
      references.push(node2);
    }
  }
  ancestor(node, {
    Pattern(node2, state, parents) {
      if (node2.type === "Identifier") {
        identifier(node2, state, parents);
      }
    },
    Identifier: identifier
  });
  return references;
}
export {
  findReferences
};
