import { mkdir, readFile, writeFile } from "node:fs/promises";
import { join } from "node:path/posix";
import { Readable } from "node:stream";
import { finished } from "node:stream/promises";
import { globSync } from "glob";
import { exports as resolveExports } from "resolve.exports";
import { rsort, satisfies } from "semver";
import { x } from "tar";
import { parseImports } from "./javascript/imports.js";
import { formatNpmSpecifier, parseNpmSpecifier } from "./npm.js";
import { initializeNpmVersionCache, resolveNpmImport, rewriteNpmImports } from "./npm.js";
import { isPathImport } from "./path.js";
import { faint } from "./tty.js";
const jsrVersionCaches = /* @__PURE__ */ new Map();
const jsrVersionRequests = /* @__PURE__ */ new Map();
const jsrPackageRequests = /* @__PURE__ */ new Map();
const jsrResolveRequests = /* @__PURE__ */ new Map();
function getJsrVersionCache(root) {
  let cache = jsrVersionCaches.get(root);
  if (!cache)
    jsrVersionCaches.set(root, cache = initializeNpmVersionCache(root, "_jsr"));
  return cache;
}
async function resolveJsrVersion(root, { name, range }) {
  const cache = await getJsrVersionCache(root);
  const versions = cache.get(name);
  if (versions) {
    for (const version of versions)
      if (!range || satisfies(version, range))
        return version;
  }
  const href = `https://npm.jsr.io/@jsr/${name.replace(/^@/, "").replace(/\//, "__")}`;
  let promise = jsrVersionRequests.get(href);
  if (promise)
    return promise;
  promise = async function() {
    process.stdout.write(`jsr:${formatNpmSpecifier({ name, range })} ${faint("\u2192")} `);
    const metaResponse = await fetch(href);
    if (!metaResponse.ok)
      throw new Error(`unable to fetch: ${href}`);
    const meta = await metaResponse.json();
    let info;
    if (meta["dist-tags"][range ?? "latest"]) {
      info = meta["versions"][meta["dist-tags"][range ?? "latest"]];
    } else if (range) {
      if (meta.versions[range]) {
        info = meta.versions[range];
      } else {
        for (const key in meta.versions) {
          if (satisfies(key, range) && !meta.versions[key].yanked) {
            info = meta.versions[key];
          }
        }
      }
    }
    if (!info)
      throw new Error(`unable to resolve version: ${formatNpmSpecifier({ name, range })}`);
    const { version, dist } = info;
    await fetchJsrPackage(root, name, version, dist.tarball);
    cache.set(name, versions ? rsort(versions.concat(version)) : [version]);
    process.stdout.write(`${version}
`);
    return version;
  }();
  promise.catch(console.error).then(() => jsrVersionRequests.delete(href));
  jsrVersionRequests.set(href, promise);
  return promise;
}
async function fetchJsrPackage(root, name, version, tarball) {
  const dir = join(root, ".observablehq", "cache", "_jsr", formatNpmSpecifier({ name, range: version }));
  let promise = jsrPackageRequests.get(dir);
  if (promise)
    return promise;
  promise = (async () => {
    const tarballResponse = await fetch(tarball);
    if (!tarballResponse.ok)
      throw new Error(`unable to fetch: ${tarball}`);
    await mkdir(dir, { recursive: true });
    await finished(Readable.fromWeb(tarballResponse.body).pipe(x({ strip: 1, C: dir })));
    await rewriteJsrImports(root, dir);
  })();
  promise.catch(console.error).then(() => jsrPackageRequests.delete(dir));
  jsrPackageRequests.set(dir, promise);
  return promise;
}
async function resolveJsrImport(root, specifier) {
  if (specifier.startsWith("/"))
    return `/_jsr/${specifier.slice("/".length)}`;
  let promise = jsrResolveRequests.get(specifier);
  if (promise)
    return promise;
  promise = async function() {
    const spec = parseNpmSpecifier(specifier);
    const { name } = spec;
    const version = await resolveJsrVersion(root, spec);
    const dir = join(root, ".observablehq", "cache", "_jsr", formatNpmSpecifier({ name, range: version }));
    const info = JSON.parse(await readFile(join(dir, "package.json"), "utf8"));
    const [path] = resolveExports(info, spec.path === void 0 ? "." : `./${spec.path}`, { browser: true });
    return join("/", "_jsr", `${name}@${version}`, path);
  }();
  jsrResolveRequests.set(specifier, promise);
  return promise;
}
async function rewriteJsrImports(root, dir) {
  const info = JSON.parse(await readFile(join(dir, "package.json"), "utf8"));
  for (const path of globSync("**/*.js", { cwd: dir, nodir: true })) {
    const input = await readFile(join(dir, path), "utf8");
    const promises = /* @__PURE__ */ new Map();
    try {
      rewriteNpmImports(input, (i) => {
        if (i.startsWith("@jsr/")) {
          const { name, path: path2 } = parseNpmSpecifier(i);
          const range = resolveDependencyVersion(info, name);
          const specifier = formatNpmSpecifier({ name: `@${name.slice("@jsr/".length).replace(/__/, "/")}`, range, path: path2 });
          if (!promises.has(i))
            promises.set(i, resolveJsrImport(root, specifier));
        } else if (!isPathImport(i) && !/^[\w-]+:/.test(i)) {
          const { name, path: path2 } = parseNpmSpecifier(i);
          const range = resolveDependencyVersion(info, i);
          const specifier = formatNpmSpecifier({ name, range, path: path2 });
          if (!promises.has(i))
            promises.set(i, resolveNpmImport(root, specifier));
        }
      });
    } catch {
      continue;
    }
    const resolutions = /* @__PURE__ */ new Map();
    for (const [key, promise] of promises)
      resolutions.set(key, await promise);
    const output = rewriteNpmImports(input, (i) => resolutions.get(i));
    await writeFile(join(dir, path), output, "utf8");
  }
}
function resolveDependencyVersion(info, name) {
  return info.dependencies?.[name] ?? info.devDependencies?.[name] ?? info.peerDependencies?.[name] ?? info.optionalDependencies?.[name] ?? info.bundleDependencies?.[name] ?? info.bundledDependencies?.[name];
}
async function resolveJsrImports(root, path) {
  if (!path.startsWith("/_jsr/"))
    throw new Error(`invalid jsr path: ${path}`);
  return parseImports(join(root, ".observablehq", "cache"), path);
}
function extractJsrSpecifier(path) {
  if (!path.startsWith("/_jsr/"))
    throw new Error(`invalid jsr path: ${path}`);
  return path.slice("/_jsr".length);
}
export {
  extractJsrSpecifier,
  resolveJsrImport,
  resolveJsrImports
};
