import { createHash } from "node:crypto";
import slugify from "@sindresorhus/slugify";
import he from "he";
import MarkdownIt from "markdown-it";
import MarkdownItAnchor from "markdown-it-anchor";
import { mergeStyle } from "./config.js";
import { readFrontMatter } from "./frontMatter.js";
import { html, rewriteHtmlPaths } from "./html.js";
import { parseInfo } from "./info.js";
import { transformJavaScriptSync } from "./javascript/module.js";
import { parseJavaScript } from "./javascript/parse.js";
import { isAssetPath, relativePath } from "./path.js";
import { parsePlaceholder } from "./placeholder.js";
import { transpileSql } from "./sql.js";
import { transpileTag } from "./tag.js";
import { InvalidThemeError } from "./theme.js";
import { red } from "./tty.js";
function uniqueCodeId(context, content) {
  const hash = createHash("sha256").update(content).digest("hex").slice(0, 8);
  let id = hash;
  let count = 1;
  while (context.code.some((code) => code.id === id))
    id = `${hash}-${count++}`;
  return id;
}
function isFalse(attribute) {
  return attribute?.toLowerCase() === "false";
}
function transpileJavaScript(content, tag) {
  try {
    return transformJavaScriptSync(content, tag);
  } catch (error) {
    throw new SyntaxError(error.message);
  }
}
function getLiveSource(content, tag, attributes) {
  return tag === "js" ? content : tag === "ts" || tag === "jsx" || tag === "tsx" ? transpileJavaScript(content, tag) : tag === "tex" ? transpileTag(content, "tex.block", true) : tag === "html" ? transpileTag(content, "html.fragment", true) : tag === "sql" ? transpileSql(content, attributes) : tag === "svg" ? transpileTag(content, "svg.fragment", true) : tag === "dot" ? transpileTag(content, "dot", false) : tag === "mermaid" ? transpileTag(content, "await mermaid", false) : void 0;
}
function makeFenceRenderer(baseRenderer) {
  return (tokens, idx, options, context, self) => {
    const { path, params } = context;
    const token = tokens[idx];
    const { tag, attributes } = parseInfo(token.info);
    token.info = tag;
    let html2 = "";
    let source;
    try {
      source = isFalse(attributes.run) ? void 0 : getLiveSource(token.content, tag, attributes);
      if (source != null) {
        const id = uniqueCodeId(context, source);
        const node = parseJavaScript(source, { path, params });
        context.code.push({ id, node, mode: tag === "jsx" || tag === "tsx" ? "jsx" : "block" });
        html2 += `<div class="observablehq observablehq--block">${node.expression ? "<observablehq-loading></observablehq-loading>" : ""}<!--:${id}:--></div>
`;
      }
    } catch (error) {
      if (!(error instanceof SyntaxError))
        throw error;
      html2 += `<div class="observablehq observablehq--block">
  <div class="observablehq--inspect observablehq--error">SyntaxError: ${he.escape(error.message)}</div>
</div>
`;
    }
    if (attributes.echo == null ? source == null : !isFalse(attributes.echo)) {
      html2 += baseRenderer(tokens, idx, options, context, self);
    }
    return html2;
  };
}
const CODE_DOLLAR = 36;
const CODE_BRACEL = 123;
const transformPlaceholderInline = (state, silent) => {
  if (silent || state.pos + 2 > state.posMax)
    return false;
  const marker1 = state.src.charCodeAt(state.pos);
  const marker2 = state.src.charCodeAt(state.pos + 1);
  if (marker1 !== CODE_DOLLAR || marker2 !== CODE_BRACEL)
    return false;
  for (const { type, content, pos } of parsePlaceholder(state.src, state.pos, state.posMax)) {
    if (type !== "placeholder")
      break;
    const token = state.push(type, "", 0);
    token.content = content;
    state.pos = pos;
    return true;
  }
  return false;
};
const transformPlaceholderCore = (state) => {
  const { tokens } = state;
  for (let i = 0, n = tokens.length; i < n; ++i) {
    const token = tokens[i];
    if (token.type === "html_block") {
      const children = [];
      for (const { type, content } of parsePlaceholder(token.content)) {
        const child = new state.Token(type, "", 0);
        child.content = content;
        children.push(child);
      }
      if (children.length === 1 && children[0].type === "html_block") {
        tokens[i].content = children[0].content;
      } else {
        const inline = new state.Token("inline", "", 0);
        inline.children = children;
        tokens[i] = inline;
      }
    }
  }
};
function makePlaceholderRenderer() {
  return (tokens, idx, options, context) => {
    const { path, params } = context;
    const token = tokens[idx];
    const id = uniqueCodeId(context, token.content);
    try {
      const node = parseJavaScript(token.content, { path, params, inline: true });
      context.code.push({ id, node, mode: "inline" });
      return `<observablehq-loading></observablehq-loading><!--:${id}:-->`;
    } catch (error) {
      if (!(error instanceof SyntaxError))
        throw error;
      return `<span class="observablehq--inspect observablehq--error" style="display: block;">SyntaxError: ${he.escape(
        error.message
      )}</span>`;
    }
  };
}
function makeSoftbreakRenderer(baseRenderer) {
  return (tokens, idx, options, context, self) => {
    context.currentLine++;
    return baseRenderer(tokens, idx, options, context, self);
  };
}
function createMarkdownIt({
  markdownIt,
  linkify = true,
  quotes = "\u201C\u201D\u2018\u2019",
  typographer = false
} = {}) {
  const md = MarkdownIt({ html: true, linkify, typographer, quotes });
  if (linkify)
    md.linkify.set({ fuzzyLink: false, fuzzyEmail: false });
  md.use(MarkdownItAnchor, { slugify: (s) => slugify(s) });
  md.inline.ruler.push("placeholder", transformPlaceholderInline);
  md.core.ruler.after("inline", "placeholder", transformPlaceholderCore);
  md.renderer.rules.placeholder = makePlaceholderRenderer();
  md.renderer.rules.fence = makeFenceRenderer(md.renderer.rules.fence);
  md.renderer.rules.softbreak = makeSoftbreakRenderer(md.renderer.rules.softbreak);
  return markdownIt === void 0 ? md : markdownIt(md);
}
function parseMarkdown(input, options) {
  const { md, path, source = path, params } = options;
  const { content, data } = readFrontMatter(input);
  const code = [];
  const context = { code, startLine: 0, currentLine: 0, path, params };
  const tokens = md.parse(content, context);
  const body = md.renderer.render(tokens, md.options, context);
  const title = data.title !== void 0 ? data.title : findTitle(tokens);
  return {
    head: getHead(title, data, options),
    header: getHeader(title, data, options),
    body,
    footer: getFooter(title, data, options),
    data,
    title,
    style: getStyle(data, options),
    code,
    path: source,
    params
  };
}
function parseMarkdownMetadata(input, options) {
  const { md, path } = options;
  const { content, data } = readFrontMatter(input);
  return {
    data,
    title: data.title !== void 0 ? data.title : findTitle(md.parse(content, { code: [], startLine: 0, currentLine: 0, path }))
  };
}
function getHead(title, data, options) {
  const { scripts, path } = options;
  let head = getHtml("head", title, data, options);
  if (scripts?.length) {
    head ??= "";
    for (const { type, async, src } of scripts) {
      head += html`${head ? "\n" : ""}<script${type ? html` type="${type}"` : null}${async ? html` async` : null} src="${isAssetPath(src) ? relativePath(path, src) : src}"></script>`;
    }
  }
  return head;
}
function getHeader(title, data, options) {
  return getHtml("header", title, data, options);
}
function getFooter(title, data, options) {
  return getHtml("footer", title, data, options);
}
function getHtml(key, title, data, { path, [key]: defaultValue }) {
  if (data[key] !== void 0)
    return data[key] != null ? String(data[key]) : null;
  const value = typeof defaultValue === "function" ? defaultValue({ title, data, path }) : defaultValue;
  return value != null ? rewriteHtmlPaths(value, path) : null;
}
function getStyle(data, { path, style = null }) {
  try {
    style = mergeStyle(path, data.style, data.theme, style);
  } catch (error) {
    if (!(error instanceof InvalidThemeError))
      throw error;
    console.error(red(String(error)));
    style = { theme: [] };
  }
  return !style ? null : "path" in style ? relativePath(path, style.path) : `observablehq:theme-${style.theme.join(",")}.css`;
}
function findTitle(tokens) {
  for (const [i, token] of tokens.entries()) {
    if (token.type === "heading_open" && token.tag === "h1") {
      const next = tokens[i + 1];
      if (next?.type === "inline") {
        const text = next.children?.filter((t) => t.type === "text").map((t) => t.content).join("");
        if (text) {
          return text;
        }
      }
    }
  }
  return null;
}
export {
  createMarkdownIt,
  parseMarkdown,
  parseMarkdownMetadata
};
