import { createHash } from "node:crypto";
import { extname, join } from "node:path/posix";
import { cacheDuckDBExtension } from "./duckdb.js";
import { findAssets } from "./html.js";
import { defaultGlobals } from "./javascript/globals.js";
import { isJavaScript } from "./javascript/imports.js";
import { getFileHash, getModuleHash, getModuleInfo } from "./javascript/module.js";
import { extractJsrSpecifier, resolveJsrImport, resolveJsrImports } from "./jsr.js";
import { getImplicitDependencies, getImplicitDownloads } from "./libraries.js";
import { getImplicitFileImports, getImplicitInputImports } from "./libraries.js";
import { getImplicitStylesheets } from "./libraries.js";
import { extractNodeSpecifier, resolveNodeImport, resolveNodeImports } from "./node.js";
import { extractNpmSpecifier, populateNpmCache, resolveNpmImport, resolveNpmImports } from "./npm.js";
import { isAssetPath, isPathImport, parseRelativeUrl } from "./path.js";
import { relativePath, resolveLocalPath, resolvePath, resolveRelativePath } from "./path.js";
const defaultImports = [
  "observablehq:client",
  // Framework client
  "observablehq:runtime",
  // Runtime
  "observablehq:stdlib"
  // Standard library
];
const builtins = /* @__PURE__ */ new Map([
  ["@observablehq/runtime", "/_observablehq/runtime.js"],
  ["@observablehq/stdlib", "/_observablehq/stdlib.js"],
  ["npm:@observablehq/runtime", "/_observablehq/runtime.js"],
  ["npm:@observablehq/stdlib", "/_observablehq/stdlib.js"],
  ["npm:@observablehq/dot", "/_observablehq/stdlib/dot.js"],
  // TODO publish to npm
  ["npm:@observablehq/duckdb", "/_observablehq/stdlib/duckdb.js"],
  // TODO publish to npm
  ["npm:@observablehq/inputs", "/_observablehq/stdlib/inputs.js"],
  // TODO publish to npm
  ["npm:@observablehq/mermaid", "/_observablehq/stdlib/mermaid.js"],
  // TODO publish to npm
  ["npm:@observablehq/tex", "/_observablehq/stdlib/tex.js"],
  // TODO publish to npm
  ["npm:@observablehq/sqlite", "/_observablehq/stdlib/sqlite.js"]
  // TODO publish to npm
]);
async function getResolvers(page, config) {
  const { path, globalStylesheets: defaultStylesheets, loaders } = config;
  const hash = createHash("sha256").update(page.body).update(JSON.stringify(page.data));
  const assets = /* @__PURE__ */ new Set();
  const files = /* @__PURE__ */ new Set();
  const fileMethods = /* @__PURE__ */ new Set();
  const anchors = /* @__PURE__ */ new Set();
  const localLinks = /* @__PURE__ */ new Set();
  const localImports = /* @__PURE__ */ new Set();
  const globalImports = new Set(defaultImports);
  const staticImports = new Set(defaultImports);
  const stylesheets = new Set(defaultStylesheets);
  for (const html of [page.head, page.header, page.body, page.footer]) {
    if (!html)
      continue;
    const info = findAssets(html, path);
    for (const f of info.files)
      assets.add(f);
    for (const a of info.anchors)
      anchors.add(a);
    for (const l of info.localLinks)
      localLinks.add(l);
    for (const i of info.localImports)
      localImports.add(i);
    for (const i of info.globalImports)
      globalImports.add(i);
    for (const i of info.staticImports)
      staticImports.add(i);
  }
  if (page.style)
    stylesheets.add(page.style);
  for (const { node } of page.code) {
    for (const f of node.files) {
      files.add(f.name);
      if (f.method)
        fileMethods.add(f.method);
    }
    for (const i of node.imports) {
      (i.type === "local" ? localImports : globalImports).add(i.name);
      if (i.method === "static")
        staticImports.add(i.name);
      if (i.type === "local" && i.method === "resolve" && !isJavaScript(i.name))
        files.add(i.name);
    }
  }
  if (page.data.sql) {
    for (const value of Object.values(page.data.sql)) {
      const source = String(value);
      if (isAssetPath(source)) {
        files.add(resolveRelativePath(path, source));
      }
    }
  }
  for (const f of assets)
    hash.update(loaders.getSourceFileHash(resolvePath(path, f)));
  for (const f of files)
    hash.update(loaders.getSourceFileHash(resolvePath(path, f)));
  for (const i of localImports)
    hash.update(loaders.getModuleHash(resolvePath(path, i)));
  if (page.style && isPathImport(page.style))
    hash.update(loaders.getSourceFileHash(resolvePath(path, page.style)));
  for (const i of getImplicitInputImports(findFreeInputs(page))) {
    staticImports.add(i);
    globalImports.add(i);
  }
  if (page.code.some((c) => c.mode === "jsx")) {
    staticImports.add("npm:react-dom");
    globalImports.add("npm:react-dom");
  }
  return {
    path,
    hash: hash.digest("hex"),
    assets,
    anchors,
    localLinks,
    ...await resolveResolvers(
      {
        files,
        fileMethods,
        localImports,
        globalImports,
        staticImports,
        stylesheets
      },
      config
    )
  };
}
async function getModuleResolvers(path, config) {
  const { root } = config;
  return {
    path,
    hash: getModuleHash(root, path),
    assets: /* @__PURE__ */ new Set(),
    anchors: /* @__PURE__ */ new Set(),
    localLinks: /* @__PURE__ */ new Set(),
    ...await resolveResolvers({ localImports: [path], staticImports: [path] }, { path, ...config })
  };
}
async function resolveResolvers({
  files: initialFiles,
  fileMethods: initialFileMethods,
  localImports: initialLocalImports,
  globalImports: initialGlobalImports,
  staticImports: intialStaticImports,
  stylesheets: initialStylesheets
}, { root, path, normalizePath, loaders, duckdb }) {
  const files = new Set(initialFiles);
  const fileMethods = new Set(initialFileMethods);
  const localImports = new Set(initialLocalImports);
  const globalImports = new Set(initialGlobalImports);
  const staticImports = new Set(intialStaticImports);
  const stylesheets = new Set(initialStylesheets);
  const resolutions = /* @__PURE__ */ new Map();
  for (const i of localImports) {
    const p = resolvePath(path, i);
    const info = getModuleInfo(root, p);
    if (!info)
      continue;
    for (const f of info.files)
      files.add(relativePath(path, resolvePath(p, f)));
    for (const m of info.fileMethods)
      fileMethods.add(m);
    for (const o of info.localStaticImports)
      localImports.add(relativePath(path, resolvePath(p, o)));
    for (const o of info.localDynamicImports)
      localImports.add(relativePath(path, resolvePath(p, o)));
    for (const o of info.globalStaticImports)
      globalImports.add(o);
    for (const o of info.globalDynamicImports)
      globalImports.add(o);
  }
  for (const i of staticImports) {
    if (!localImports.has(i))
      continue;
    const p = resolvePath(path, i);
    const info = getModuleInfo(root, p);
    if (!info)
      continue;
    for (const o of info.localStaticImports)
      staticImports.add(relativePath(path, resolvePath(p, o)));
    for (const o of info.globalStaticImports)
      staticImports.add(o);
  }
  for (const i of getImplicitFileImports(fileMethods)) {
    staticImports.add(i);
    globalImports.add(i);
  }
  for (const i of getImplicitDependencies(staticImports)) {
    staticImports.add(i);
  }
  for (const i of getImplicitDependencies(globalImports)) {
    globalImports.add(i);
  }
  for (const i of globalImports) {
    if (builtins.has(i))
      continue;
    if (i.startsWith("npm:")) {
      resolutions.set(i, await resolveNpmImport(root, i.slice("npm:".length)));
    } else if (i.startsWith("jsr:")) {
      resolutions.set(i, await resolveJsrImport(root, i.slice("jsr:".length)));
    } else if (!/^\w+:/.test(i)) {
      try {
        resolutions.set(i, await resolveNodeImport(root, i));
      } catch {
      }
    }
  }
  for (const [key, value] of resolutions) {
    if (key.startsWith("npm:")) {
      for (const i of await resolveNpmImports(root, value)) {
        if (i.type === "local") {
          const path2 = resolvePath(value, i.name);
          const specifier = `npm:${extractNpmSpecifier(path2)}`;
          globalImports.add(specifier);
          resolutions.set(specifier, path2);
        }
      }
    } else if (key.startsWith("jsr:")) {
      for (const i of await resolveJsrImports(root, value)) {
        if (i.type === "local") {
          const path2 = resolvePath(value, i.name);
          let specifier;
          if (path2.startsWith("/_npm/"))
            specifier = `npm:${extractNpmSpecifier(path2)}`;
          else if (path2.startsWith("/_jsr/"))
            specifier = `jsr:${extractJsrSpecifier(path2)}`;
          else
            continue;
          globalImports.add(specifier);
          resolutions.set(specifier, path2);
        }
      }
    } else if (!/^\w+:/.test(key)) {
      for (const i of await resolveNodeImports(root, value)) {
        if (i.type === "local") {
          const path2 = resolvePath(value, i.name);
          const specifier = extractNodeSpecifier(path2);
          globalImports.add(specifier);
          resolutions.set(specifier, path2);
        }
      }
    }
  }
  const staticResolutions = /* @__PURE__ */ new Map();
  for (const i of staticImports) {
    if (i.startsWith("npm:") || i.startsWith("jsr:") || !/^\w+:/.test(i)) {
      const r = resolutions.get(i);
      if (r)
        staticResolutions.set(i, r);
    }
  }
  for (const [key, value] of staticResolutions) {
    if (key.startsWith("npm:")) {
      for (const i of await resolveNpmImports(root, value)) {
        if (i.type === "local" && i.method === "static") {
          const path2 = resolvePath(value, i.name);
          const specifier = `npm:${extractNpmSpecifier(path2)}`;
          staticImports.add(specifier);
          staticResolutions.set(specifier, path2);
        }
      }
    } else if (key.startsWith("jsr:")) {
      for (const i of await resolveJsrImports(root, value)) {
        if (i.type === "local" && i.method === "static") {
          const path2 = resolvePath(value, i.name);
          let specifier;
          if (path2.startsWith("/_npm/"))
            specifier = `npm:${extractNpmSpecifier(path2)}`;
          else if (path2.startsWith("/_jsr/"))
            specifier = `jsr:${extractJsrSpecifier(path2)}`;
          else
            continue;
          staticImports.add(specifier);
          staticResolutions.set(specifier, path2);
        }
      }
    } else if (!/^\w+:/.test(key)) {
      for (const i of await resolveNodeImports(root, value)) {
        if (i.type === "local" && i.method === "static") {
          const path2 = resolvePath(value, i.name);
          const specifier = extractNodeSpecifier(path2);
          staticImports.add(specifier);
          staticResolutions.set(specifier, path2);
        }
      }
    }
  }
  for (const specifier of getImplicitStylesheets(staticImports)) {
    stylesheets.add(specifier);
    if (specifier.startsWith("npm:")) {
      const path2 = await resolveNpmImport(root, specifier.slice("npm:".length));
      resolutions.set(specifier, path2);
      await populateNpmCache(root, path2);
    } else if (!specifier.startsWith("observablehq:")) {
      throw new Error(`unhandled implicit stylesheet: ${specifier}`);
    }
  }
  for (const specifier of getImplicitDownloads(globalImports, duckdb)) {
    globalImports.add(specifier);
    if (specifier.startsWith("npm:")) {
      const path2 = await resolveNpmImport(root, specifier.slice("npm:".length));
      resolutions.set(specifier, path2);
      await populateNpmCache(root, path2);
    } else if (specifier.startsWith("duckdb:")) {
      const path2 = await cacheDuckDBExtension(root, specifier.slice("duckdb:".length));
      resolutions.set(specifier, path2);
    } else if (!specifier.startsWith("observablehq:")) {
      throw new Error(`unhandled implicit download: ${specifier}`);
    }
  }
  function resolveImport(specifier) {
    return isPathImport(specifier) ? relativePath(path, loaders.resolveImportPath(resolvePath(path, specifier))) : builtins.has(specifier) ? relativePath(path, builtins.get(specifier)) : specifier.startsWith("observablehq:") ? relativePath(path, `/_observablehq/${specifier.slice("observablehq:".length)}${extname(specifier) ? "" : ".js"}`) : resolutions.has(specifier) ? relativePath(path, resolutions.get(specifier)) : specifier;
  }
  function resolveFile(specifier) {
    return relativePath(path, loaders.resolveFilePath(resolvePath(path, specifier)));
  }
  function resolveStylesheet(specifier) {
    return isPathImport(specifier) ? relativePath(path, resolveStylesheetPath(root, resolvePath(path, specifier))) : specifier.startsWith("observablehq:") ? relativePath(path, `/_observablehq/${specifier.slice("observablehq:".length)}`) : resolutions.has(specifier) ? relativePath(path, resolutions.get(specifier)) : specifier;
  }
  function resolveScript(src) {
    if (isAssetPath(src)) {
      const localPath = resolveLocalPath(path, src);
      return localPath ? resolveImport(relativePath(path, localPath)) : src;
    } else {
      return resolveImport(src);
    }
  }
  function resolveLink(href) {
    if (isAssetPath(href)) {
      const u = parseRelativeUrl(href);
      const localPath = resolveLocalPath(path, u.pathname);
      if (localPath)
        return relativePath(path, normalizePath(localPath)) + u.search + u.hash;
    }
    return href;
  }
  return {
    files,
    localImports,
    globalImports,
    staticImports,
    stylesheets,
    resolveFile,
    resolveImport,
    resolveScript,
    resolveStylesheet,
    resolveLink
  };
}
async function getModuleStaticImports(root, path) {
  const localImports = /* @__PURE__ */ new Set([path]);
  const globalImports = /* @__PURE__ */ new Set();
  const fileMethods = /* @__PURE__ */ new Set();
  for (const i of localImports) {
    const info = getModuleInfo(root, i);
    if (!info)
      continue;
    for (const o of info.localStaticImports)
      localImports.add(resolvePath(i, o));
    for (const o of info.globalStaticImports)
      globalImports.add(o);
    for (const m of info.fileMethods)
      fileMethods.add(m);
  }
  for (const i of getImplicitFileImports(fileMethods)) {
    globalImports.add(i);
  }
  for (const i of globalImports) {
    if (builtins.has(i))
      continue;
    if (i.startsWith("npm:")) {
      const p = await resolveNpmImport(root, i.slice("npm:".length));
      for (const o of await resolveNpmImports(root, p)) {
        if (o.type === "local")
          globalImports.add(`npm:${extractNpmSpecifier(resolvePath(p, o.name))}`);
      }
    } else if (i.startsWith("jsr:")) {
      const p = await resolveJsrImport(root, i.slice("jsr:".length));
      for (const o of await resolveJsrImports(root, p)) {
        if (o.type === "local")
          globalImports.add(`jsr:${extractJsrSpecifier(resolvePath(p, o.name))}`);
      }
    } else if (!/^\w+:/.test(i)) {
      const p = await resolveNodeImport(root, i);
      for (const o of await resolveNodeImports(root, p)) {
        if (o.type === "local")
          globalImports.add(extractNodeSpecifier(resolvePath(p, o.name)));
      }
    }
  }
  return [...localImports, ...globalImports].filter((i) => i !== path).map((i) => relativePath(path, i));
}
function getModuleResolver(root, path, servePath = `/${join("_import", path)}`, getHash) {
  return async (specifier) => {
    return isPathImport(specifier) ? relativePath(servePath, resolveImportPath(root, resolvePath(path, specifier), getHash)) : builtins.has(specifier) || specifier.startsWith("observablehq:") ? relativePath(servePath, resolveBuiltin(specifier)) : specifier.startsWith("npm:") ? relativePath(servePath, await resolveNpmImport(root, specifier.slice("npm:".length))) : specifier.startsWith("jsr:") ? relativePath(servePath, await resolveJsrImport(root, specifier.slice("jsr:".length))) : !/^\w+:/.test(specifier) ? relativePath(servePath, await resolveNodeImport(root, specifier)) : specifier;
  };
}
function resolveBuiltin(specifier) {
  if (builtins.has(specifier))
    return builtins.get(specifier);
  if (!specifier.startsWith("observablehq:"))
    throw new Error(`not built-in: ${specifier}`);
  return `/_observablehq/${specifier.slice("observablehq:".length)}${extname(specifier) ? "" : ".js"}`;
}
function resolveStylesheetPath(root, path) {
  return `/${join("_import", path)}?sha=${getFileHash(root, path)}`;
}
function resolveImportPath(root, path, getHash) {
  return `/${join("_import", path)}?sha=${getModuleHash(root, path, getHash)}`;
}
function findFreeInputs(page) {
  const outputs = new Set(defaultGlobals).add("display").add("view").add("visibility").add("invalidation");
  const inputs = /* @__PURE__ */ new Set();
  for (const { node } of page.code) {
    if (node.declarations) {
      for (const { name } of node.declarations) {
        outputs.add(name);
      }
    }
  }
  for (const { node } of page.code) {
    for (const { name } of node.references) {
      if (!outputs.has(name)) {
        inputs.add(name);
      }
    }
  }
  return inputs;
}
export {
  builtins,
  getModuleResolver,
  getModuleResolvers,
  getModuleStaticImports,
  getResolvers,
  resolveBuiltin,
  resolveImportPath,
  resolveStylesheetPath
};
