import { extname, resolve } from "node:path/posix";
import { nodeResolve } from "@rollup/plugin-node-resolve";
import { simple } from "acorn-walk";
import { build } from "esbuild";
import { rollup } from "rollup";
import esbuild from "rollup-plugin-esbuild";
import { getClientPath, getStylePath } from "./files.js";
import { annotatePath } from "./javascript/annotate.js";
import { getStringLiteralValue, isStringLiteral } from "./javascript/source.js";
import { resolveNpmImport } from "./npm.js";
import { getObservableUiOrigin } from "./observableApiClient.js";
import { isAssetPath, isPathImport, relativePath } from "./path.js";
import { builtins } from "./resolvers.js";
import { Sourcemap } from "./sourcemap.js";
import { THEMES, renderTheme } from "./theme.js";
const STYLE_MODULES = {
  "observablehq:default.css": getStylePath("default.css"),
  ...Object.fromEntries(THEMES.map(({ name, path }) => [`observablehq:theme-${name}.css`, path]))
};
const BUNDLED_MODULES = [
  "@observablehq/inputs",
  // observablehq:stdlib/inputs.js
  "@observablehq/inspector",
  // observablehq:runtime.js
  "@observablehq/runtime",
  // observablehq:runtime.js
  "isoformat",
  // observablehq:runtime.js
  "minisearch"
  // observablehq:search.js
];
function rewriteInputsNamespace(code) {
  return code.replace(/\b__ns__\b/g, "inputs-3a86ea");
}
async function bundleStyles({
  minify = false,
  path,
  theme
}) {
  const result = await build({
    bundle: true,
    ...path ? { entryPoints: [path] } : { stdin: { contents: renderTheme(theme), loader: "css" } },
    write: false,
    minify,
    alias: STYLE_MODULES
  });
  let text = result.outputFiles[0].text;
  if (path === getClientPath("stdlib/inputs.css"))
    text = rewriteInputsNamespace(text);
  return text;
}
async function rollupClient(input, root, path, {
  define,
  keepNames,
  minify,
  resolveImport: resolveImport2 = getDefaultResolver(root)
} = {}) {
  if (typeof resolveImport2 !== "function")
    throw new Error(`invalid resolveImport: ${resolveImport2}`);
  const bundle = await rollup({
    input,
    external: [/^https:/],
    plugins: [
      nodeResolve({ resolveOnly: BUNDLED_MODULES }),
      importResolve(input, path, resolveImport2),
      esbuild({
        format: "esm",
        platform: "browser",
        target: ["es2022", "chrome96", "firefox96", "safari16", "node18"],
        exclude: [],
        // don’t exclude node_modules
        keepNames,
        minify,
        define: {
          "process.env.OBSERVABLE_ORIGIN": JSON.stringify(String(getObservableUiOrigin()).replace(/\/$/, "")),
          ...define
        }
      }),
      importMetaResolve(path, resolveImport2)
    ],
    onwarn(message, warn) {
      if (message.code === "CIRCULAR_DEPENDENCY")
        return;
      warn(message);
    }
  });
  try {
    const output = await bundle.generate({ format: "es" });
    const code = output.output.find((o) => o.type === "chunk").code;
    return rewriteTypeScriptImports(code);
  } finally {
    await bundle.close();
  }
}
function rewriteTypeScriptImports(code) {
  return code.replace(/(?<=\bimport\(([`'"])[\w./]+)\.ts(?=\1\))/g, ".js");
}
function getDefaultResolver(root) {
  return (specifier) => resolveImport(root, specifier);
}
async function resolveImport(root, specifier) {
  return BUNDLED_MODULES.includes(specifier) ? void 0 : builtins.has(specifier) ? builtins.get(specifier) : specifier.startsWith("observablehq:") ? `/_observablehq/${specifier.slice("observablehq:".length)}${extname(specifier) ? "" : ".js"}` : specifier.startsWith("npm:") ? await resolveNpmImport(root, specifier.slice("npm:".length)) : !/^[a-z]:\\/i.test(specifier) && !isPathImport(specifier) ? await resolveNpmImport(root, specifier) : void 0;
}
function importResolve(input, path, resolveImport2) {
  input = resolve(input);
  async function resolveId(specifier) {
    if (typeof specifier !== "string")
      return null;
    if (isAssetPath(specifier) && resolve(specifier) === input)
      return null;
    const resolution = await resolveImport2(specifier);
    if (resolution)
      return { id: relativePath(path, resolution), external: true };
    return null;
  }
  return {
    name: "resolve-import",
    resolveId,
    resolveDynamicImport: resolveId
  };
}
function importMetaResolve(path, resolveImport2) {
  return {
    name: "resolve-import-meta-resolve",
    async transform(code) {
      const program = this.parse(code);
      const resolves = [];
      simple(program, {
        CallExpression(node) {
          if (node.callee.type === "MemberExpression" && node.callee.object.type === "MetaProperty" && node.callee.property.type === "Identifier" && node.callee.property.name === "resolve" && node.arguments.length === 1 && isStringLiteral(node.arguments[0])) {
            resolves.push(node.arguments[0]);
          }
        }
      });
      if (!resolves.length)
        return null;
      const output = new Sourcemap(code);
      for (const source of resolves) {
        const specifier = getStringLiteralValue(source);
        const resolution = await resolveImport2(specifier);
        if (resolution)
          output.replaceLeft(source.start, source.end, annotatePath(relativePath(path, resolution)));
      }
      return { code: String(output) };
    }
  };
}
export {
  bundleStyles,
  resolveImport,
  rollupClient
};
