import { existsSync, statSync } from "node:fs";
import { basename, join } from "node:path/posix";
import { globSync } from "glob";
function isParameterized(name) {
  return /\[([a-z_]\w*)\]/i.test(name);
}
function route(root, path, exts) {
  return routeParams(root, ".", join(".", path).split("/"), exts);
}
function routeParams(root, cwd, parts, exts) {
  switch (parts.length) {
    case 0:
      return;
    case 1: {
      const [first] = parts;
      if (!isParameterized(first)) {
        for (const ext of exts) {
          const path = join(root, cwd, first + ext);
          if (existsSync(path)) {
            if (!statSync(path).isFile())
              return;
            return { path: join(cwd, first + ext), ext };
          }
        }
      }
      if (first) {
        for (const ext of exts) {
          for (const file of globSync(`*\\[?*\\]*${ext}`, { cwd: join(root, cwd), nodir: true })) {
            const params = matchParams(basename(file, ext), first);
            if (params)
              return { path: join(cwd, file), params: { ...params }, ext };
          }
        }
      }
      return;
    }
    default: {
      const [first, ...rest] = parts;
      const path = join(root, cwd, first);
      if (existsSync(path)) {
        if (!statSync(path).isDirectory())
          return;
        if (!isParameterized(first)) {
          const found = routeParams(root, join(cwd, first), rest, exts);
          if (found)
            return found;
        }
      }
      if (first) {
        for (const dir of globSync("*\\[?*\\]*/", { cwd: join(root, cwd) })) {
          const params = matchParams(dir, first);
          if (!params)
            continue;
          const found = routeParams(root, join(cwd, dir), rest, exts);
          if (found)
            return { ...found, params: { ...found.params, ...params } };
        }
      }
    }
  }
}
function matchParams(file, input) {
  return compilePattern(file).exec(input)?.groups;
}
function compilePattern(file) {
  let pattern = "^";
  let i = 0;
  for (let match, re = /\[([a-z_]\w*)\]/gi; match = re.exec(file); i = re.lastIndex) {
    pattern += `${requote(file.slice(i, match.index))}(?<${match[1]}>.+)`;
  }
  pattern += `${requote(file.slice(i))}$`;
  return new RegExp(pattern, "i");
}
function requote(text) {
  return text.replace(/[\\^$*+?|[\]().{}]/g, "\\$&");
}
export {
  isParameterized,
  requote,
  route
};
