import { exec } from "node:child_process";
import { createHash, randomUUID } from "node:crypto";
import { readFile, writeFile } from "node:fs/promises";
import os from "node:os";
import { join } from "node:path/posix";
import { CliError } from "./error.js";
import { getObservableUiOrigin } from "./observableApiClient.js";
import { link, magenta } from "./tty.js";
const defaultEffects = {
  logger: console,
  process,
  readFile,
  writeFile
};
function getOrigin(env) {
  const urlText = env["OBSERVABLE_TELEMETRY_ORIGIN"];
  if (urlText) {
    try {
      return new URL(urlText);
    } catch (error) {
      throw new CliError(`Invalid OBSERVABLE_TELEMETRY_ORIGIN: ${urlText}`, { cause: error });
    }
  }
  const origin = getObservableUiOrigin(env);
  origin.hostname = "events." + origin.hostname;
  return origin;
}
class Telemetry {
  effects;
  disabled;
  debug;
  endpoint;
  timeZoneOffset = (/* @__PURE__ */ new Date()).getTimezoneOffset();
  _pending = /* @__PURE__ */ new Set();
  _config;
  _ids;
  _environment;
  static _instance;
  static get instance() {
    return this._instance ??= new Telemetry();
  }
  static record(data) {
    return Telemetry.instance.record(data);
  }
  constructor(effects = defaultEffects) {
    this.effects = effects;
    const { process: process2 } = effects;
    this.disabled = !!process2.env.OBSERVABLE_TELEMETRY_DISABLE;
    this.debug = !!process2.env.OBSERVABLE_TELEMETRY_DEBUG;
    this.endpoint = new URL("/cli", getOrigin(process2.env));
    this.handleSignal("SIGHUP");
    this.handleSignal("SIGINT");
    this.handleSignal("SIGTERM");
  }
  record(data) {
    if (this.disabled)
      return;
    const task = (async () => this.send({
      ids: await this.ids,
      environment: await this.environment,
      time: { now: performance.now(), timeOrigin: performance.timeOrigin, timeZoneOffset: this.timeZoneOffset },
      data
    }).catch(() => {
    }).finally(() => {
      this._pending.delete(task);
    }))();
    this._pending.add(task);
  }
  get pending() {
    return Promise.all(this._pending);
  }
  handleSignal(name) {
    const { process: process2 } = this.effects;
    let exiting = false;
    const signaled = async (signal) => {
      if (exiting)
        return;
      exiting = true;
      this.record({ event: "signal", signal });
      try {
        await Promise.race([this.pending, new Promise((resolve) => setTimeout(resolve, 1e3))]);
      } catch {
      }
      process2.off(name, signaled);
      process2.kill(process2.pid, signal);
    };
    process2.on(name, signaled);
  }
  async getPersistentId(name, generator = randomUUID) {
    const { readFile: readFile2, writeFile: writeFile2 } = this.effects;
    const file = join(os.homedir(), ".observablehq");
    if (!this._config) {
      this._config = readFile2(file, "utf8").then(JSON.parse).catch(() => ({}));
    }
    const config = await this._config;
    if (!config[name]) {
      config[name] = generator();
      try {
        await writeFile2(file, JSON.stringify(config, null, 2));
      } catch {
        return null;
      }
    }
    return config[name];
  }
  async getProjectId() {
    const salt = await this.getPersistentId("cli_telemetry_salt");
    if (!salt)
      return null;
    const remote = await new Promise((resolve) => {
      exec("git config --local --get remote.origin.url", (error, stdout) => resolve(error ? null : stdout.trim()));
    });
    const hash = createHash("sha256");
    hash.update(salt);
    hash.update(remote || this.effects.process.env.REPOSITORY_URL || process.cwd());
    return hash.digest("base64");
  }
  get ids() {
    return this._ids ??= Promise.all([this.getPersistentId("cli_telemetry_device"), this.getProjectId()]).then(
      ([device, project]) => {
        const ids = {
          session: randomUUID(),
          device,
          project
        };
        return ids;
      }
    );
  }
  get environment() {
    return this._environment ??= Promise.all([import("ci-info"), import("is-docker"), import("is-wsl")]).then(
      ([ci, { default: isDocker }, { default: isWSL }]) => {
        const cpus = os.cpus() || [];
        const environment = {
          version: "1.13.3",
          userAgent: process.env.npm_config_user_agent,
          node: process.versions.node,
          systemPlatform: os.platform(),
          systemRelease: os.release(),
          systemArchitecture: os.arch(),
          cpuCount: cpus.length,
          cpuModel: cpus.length ? cpus[0].model : null,
          cpuSpeed: cpus.length ? cpus[0].speed : null,
          memoryInMb: Math.trunc(os.totalmem() / Math.pow(1024, 2)),
          isCI: ci.name || ci.isCI,
          isDocker: isDocker(),
          isWSL
        };
        return environment;
      }
    );
  }
  async showBannerIfNeeded() {
    let called;
    await this.getPersistentId("cli_telemetry_banner", () => called = randomUUID());
    if (called) {
      this.effects.logger.error(
        `
${magenta("Attention:")} Observable Framework collects anonymous telemetry to help us improve
           the product. See ${link("https://observablehq.com/framework/telemetry")} for details.
           Set \`OBSERVABLE_TELEMETRY_DISABLE=true\` to disable.`
      );
    }
  }
  async send(data) {
    await this.showBannerIfNeeded();
    if (this.debug) {
      this.effects.logger.error("[telemetry]", data);
      return;
    }
    await fetch(this.endpoint, {
      method: "POST",
      body: JSON.stringify(data),
      headers: { "content-type": "application/json" }
    });
  }
}
export {
  Telemetry
};
