import { existsSync, readdirSync, statSync } from "node:fs";
import { mkdir, stat } from "node:fs/promises";
import op from "node:path";
import { join, normalize, relative, sep } from "node:path/posix";
import { cwd } from "node:process";
import { fileURLToPath } from "node:url";
import { isEnoent } from "./error.js";
const toOsPath = sep === op.sep ? (path) => path : (path) => path.split(sep).join(op.sep);
const fromOsPath = sep === op.sep ? (path) => path : (path) => path.split(op.sep).join(sep);
function getClientPath(entry) {
  const path = fromOsPath(op.relative(cwd(), op.join(fileURLToPath(import.meta.url), "..", "client", entry)));
  if (path.endsWith(".js") && !existsSync(path)) {
    const tspath = path.slice(0, -".js".length) + ".ts";
    if (existsSync(tspath))
      return tspath;
  }
  return path;
}
function getStylePath(entry) {
  return fromOsPath(op.relative(cwd(), op.join(fileURLToPath(import.meta.url), "..", "style", entry)));
}
function* visitFiles(root, test) {
  const visited = /* @__PURE__ */ new Set();
  const queue = [root = normalize(root)];
  for (const path of queue) {
    try {
      const status = statSync(path);
      if (status.isDirectory()) {
        if (visited.has(status.ino))
          continue;
        visited.add(status.ino);
        for (const entry of readdirSync(path)) {
          if (entry === ".observablehq")
            continue;
          if (test !== void 0 && !test(entry))
            continue;
          queue.push(join(path, entry));
        }
      } else {
        yield relative(root, path);
      }
    } catch (error) {
      if (!isEnoent(error))
        throw error;
    }
  }
}
async function maybeStat(path) {
  try {
    return await stat(path);
  } catch (error) {
    if (!isEnoent(error))
      throw error;
  }
}
async function prepareOutput(outputPath) {
  const outputDir = op.dirname(outputPath);
  if (outputDir === ".")
    return;
  await mkdir(outputDir, { recursive: true });
}
export {
  fromOsPath,
  getClientPath,
  getStylePath,
  maybeStat,
  prepareOutput,
  toOsPath,
  visitFiles
};
