import { createHash } from "node:crypto";
import { accessSync, constants, readFileSync, statSync } from "node:fs";
import { readFile } from "node:fs/promises";
import { extname, join } from "node:path/posix";
import { transform, transformSync } from "esbuild";
import { currentDate } from "../config.js";
import { resolveJsrImport } from "../jsr.js";
import { resolveNodeImport } from "../node.js";
import { resolveNpmImport } from "../npm.js";
import { resolvePath } from "../path.js";
import { builtins, resolveBuiltin } from "../resolvers.js";
import { route } from "../route.js";
import { findFiles } from "./files.js";
import { findImports, parseImports } from "./imports.js";
import { parseProgram } from "./parse.js";
const fileInfoCache = /* @__PURE__ */ new Map();
const moduleInfoCache = /* @__PURE__ */ new Map();
function getModuleHash(root, path, getHash) {
  return getModuleHashInternal(root, path, getHash).digest("hex");
}
function getModuleHashInternal(root, path, getHash = (p) => getFileHash(root, p)) {
  const hash = createHash("sha256");
  const paths = /* @__PURE__ */ new Set([path]);
  for (const path2 of paths) {
    if (path2.endsWith(".js")) {
      const info = getModuleInfo(root, path2);
      if (!info)
        continue;
      hash.update(info.hash);
      for (const i of info.localStaticImports) {
        paths.add(resolvePath(path2, i));
      }
      for (const i of info.localDynamicImports) {
        paths.add(resolvePath(path2, i));
      }
      for (const i of info.files) {
        hash.update(getHash(resolvePath(path2, i)));
      }
    } else {
      hash.update(getHash(path2));
    }
  }
  return hash;
}
async function getLocalModuleHash(root, path, getHash) {
  const hash = getModuleHashInternal(root, path, getHash);
  const info = getModuleInfo(root, path);
  if (info) {
    const globalPaths = /* @__PURE__ */ new Set();
    for (const i of [...info.globalStaticImports, ...info.globalDynamicImports]) {
      if (builtins.has(i) || i.startsWith("observablehq:")) {
        hash.update(`${resolveBuiltin(i)}?version=${"1.13.3"}`);
      } else if (i.startsWith("npm:")) {
        globalPaths.add(await resolveNpmImport(root, i.slice("npm:".length)));
      } else if (i.startsWith("jsr:")) {
        globalPaths.add(await resolveJsrImport(root, i.slice("jsr:".length)));
      } else if (!/^\w+:/.test(i)) {
        globalPaths.add(await resolveNodeImport(root, i));
      }
    }
    for (const p of globalPaths) {
      hash.update(p);
      for (const i of await parseImports(join(root, ".observablehq", "cache"), p)) {
        if (i.type === "local") {
          globalPaths.add(resolvePath(p, i.name));
        }
      }
    }
  }
  return hash.digest("hex");
}
function getModuleInfo(root, path) {
  const module = findModule(root, path);
  if (!module)
    return;
  const key = join(root, module.path);
  let mtimeMs;
  try {
    mtimeMs = Math.floor(statSync(key).mtimeMs);
  } catch {
    moduleInfoCache.delete(key);
    return;
  }
  let info = moduleInfoCache.get(key);
  if (!info || info.mtimeMs < mtimeMs) {
    let source;
    let body;
    try {
      source = readJavaScriptSync(key);
      body = parseProgram(source, module.params);
    } catch {
      moduleInfoCache.delete(key);
      return;
    }
    const hash = createHash("sha256").update(source).digest("hex");
    const imports = findImports(body, path, source);
    const files = findFiles(body, path, source);
    const localStaticImports = /* @__PURE__ */ new Set();
    const localDynamicImports = /* @__PURE__ */ new Set();
    const globalStaticImports = /* @__PURE__ */ new Set();
    const globalDynamicImports = /* @__PURE__ */ new Set();
    for (const i of imports) {
      (i.type === "local" ? i.method === "static" ? localStaticImports : localDynamicImports : i.method === "static" ? globalStaticImports : globalDynamicImports).add(i.name);
    }
    moduleInfoCache.set(
      key,
      info = {
        mtimeMs,
        hash,
        files: new Set(files.map((f) => f.name)),
        fileMethods: new Set(files.map((f) => f.method).filter((m) => m !== void 0)),
        localStaticImports,
        localDynamicImports,
        globalStaticImports,
        globalDynamicImports
      }
    );
  }
  return info;
}
function getFileHash(root, path) {
  return getFileInfo(root, path)?.hash ?? createHash("sha256").digest("hex");
}
function getFileInfo(root, path) {
  const key = join(root, path);
  let mtimeMs;
  let size;
  try {
    const stat = statSync(key);
    if (!stat.isFile())
      return;
    accessSync(key, constants.R_OK);
    mtimeMs = Math.floor(currentDate ?? stat.mtimeMs);
    size = stat.size;
  } catch {
    fileInfoCache.delete(key);
    return;
  }
  let entry = fileInfoCache.get(key);
  if (!entry || entry.mtimeMs < mtimeMs) {
    const contents = readFileSync(key);
    const hash = createHash("sha256").update(contents).digest("hex");
    fileInfoCache.set(key, entry = { mtimeMs, size, hash });
  }
  return entry;
}
function clearFileInfo(root, path) {
  return fileInfoCache.delete(join(root, path));
}
function findModule(root, path) {
  const ext = extname(path);
  if (!ext)
    throw new Error(`empty extension: ${path}`);
  const exts = [ext];
  if (ext === ".js")
    exts.push(".ts", ".jsx", ".tsx");
  return route(root, path.slice(0, -ext.length), exts);
}
async function readJavaScript(sourcePath) {
  const source = await readFile(sourcePath, "utf-8");
  switch (extname(sourcePath)) {
    case ".ts":
      return transformJavaScript(source, "ts", sourcePath);
    case ".jsx":
      return transformJavaScript(source, "jsx", sourcePath);
    case ".tsx":
      return transformJavaScript(source, "tsx", sourcePath);
  }
  return source;
}
function readJavaScriptSync(sourcePath) {
  const source = readFileSync(sourcePath, "utf-8");
  switch (extname(sourcePath)) {
    case ".ts":
      return transformJavaScriptSync(source, "ts", sourcePath);
    case ".jsx":
      return transformJavaScriptSync(source, "jsx", sourcePath);
    case ".tsx":
      return transformJavaScriptSync(source, "tsx", sourcePath);
  }
  return source;
}
async function transformJavaScript(source, loader, sourcePath) {
  return (await transform(source, getTransformOptions(loader, sourcePath))).code;
}
function transformJavaScriptSync(source, loader, sourcePath) {
  return transformSync(source, getTransformOptions(loader, sourcePath)).code;
}
function getTransformOptions(loader, sourcePath) {
  switch (loader) {
    case "ts":
      return {
        loader,
        sourcefile: sourcePath,
        tsconfigRaw: '{"compilerOptions": {"verbatimModuleSyntax": true}}'
      };
    case "jsx":
      return {
        loader,
        jsx: "automatic",
        jsxImportSource: "npm:react",
        sourcefile: sourcePath
      };
    case "tsx":
      return {
        loader,
        jsx: "automatic",
        jsxImportSource: "npm:react",
        sourcefile: sourcePath,
        tsconfigRaw: '{"compilerOptions": {"verbatimModuleSyntax": true}}'
      };
    default:
      throw new Error(`unknown loader: ${loader}`);
  }
}
export {
  clearFileInfo,
  findModule,
  getFileHash,
  getFileInfo,
  getLocalModuleHash,
  getModuleHash,
  getModuleInfo,
  readJavaScript,
  readJavaScriptSync,
  transformJavaScript,
  transformJavaScriptSync
};
