import { existsSync } from "node:fs";
import { mkdir, readFile, readdir, writeFile } from "node:fs/promises";
import { dirname, extname, join } from "node:path/posix";
import { simple } from "acorn-walk";
import { maxSatisfying, rsort, satisfies, validRange } from "semver";
import { DUCKDB_WASM_VERSION } from "./duckdb.js";
import { isEnoent } from "./error.js";
import { annotatePath } from "./javascript/annotate.js";
import { isImportMetaResolve, parseImports } from "./javascript/imports.js";
import { parseProgram } from "./javascript/parse.js";
import { getStringLiteralValue, isStringLiteral } from "./javascript/source.js";
import { relativePath } from "./path.js";
import { Sourcemap } from "./sourcemap.js";
import { faint, yellow } from "./tty.js";
function parseNpmSpecifier(specifier) {
  const parts = specifier.split("/");
  const namerange = specifier.startsWith("@") ? [parts.shift(), parts.shift()].join("/") : parts.shift();
  const ranged = namerange.indexOf("@", 1);
  return {
    name: ranged > 0 ? namerange.slice(0, ranged) : namerange,
    range: ranged > 0 ? namerange.slice(ranged + 1) : void 0,
    path: parts.length > 0 ? parts.join("/") : void 0
  };
}
function formatNpmSpecifier({ name, range, path }) {
  return `${name}${range ? `@${range}` : ""}${path ? `/${path}` : ""}`;
}
function rewriteNpmImports(input, resolve = () => void 0) {
  const body = parseProgram(input);
  const output = new Sourcemap(input);
  simple(body, {
    ImportDeclaration: rewriteImport,
    ImportExpression: rewriteImport,
    ExportAllDeclaration: rewriteImport,
    ExportNamedDeclaration: rewriteImport,
    CallExpression: rewriteImportMetaResolve
  });
  function rewriteImport(node) {
    if (node.source && isStringLiteral(node.source)) {
      rewriteImportSource(node.source);
    }
  }
  function rewriteImportMetaResolve(node) {
    if (isImportMetaResolve(node) && isStringLiteral(node.arguments[0])) {
      rewriteImportSource(node.arguments[0]);
    }
  }
  function rewriteImportSource(source) {
    const value = getStringLiteralValue(source);
    const resolved = resolve(value);
    if (resolved === void 0 || value === resolved)
      return;
    output.replaceLeft(source.start, source.end, annotatePath(resolved));
  }
  return String(output).replace(/^\/\/# sourceMappingURL=.*$\n?/m, "");
}
const npmRequests = /* @__PURE__ */ new Map();
async function populateNpmCache(root, path) {
  if (!path.startsWith("/_npm/"))
    throw new Error(`invalid npm path: ${path}`);
  const outputPath = join(root, ".observablehq", "cache", path);
  if (existsSync(outputPath))
    return outputPath;
  let promise = npmRequests.get(outputPath);
  if (promise)
    return promise;
  promise = (async () => {
    let specifier = extractNpmSpecifier(path);
    const s = parseNpmSpecifier(specifier);
    if (s.name === "sql.js" && s.path === "+esm") {
      specifier = formatNpmSpecifier({ ...s, path: "dist/sql-wasm.js" });
    }
    const href = `https://cdn.jsdelivr.net/npm/${specifier}`;
    console.log(`npm:${specifier} ${faint("\u2192")} ${outputPath}`);
    const response = await fetch(href);
    if (!response.ok)
      throw new Error(`unable to fetch: ${href}`);
    await mkdir(dirname(outputPath), { recursive: true });
    if (/^application\/javascript(;|$)/i.test(response.headers.get("content-type"))) {
      let source = await response.text();
      if (s.name === "sql.js" && s.path === "+esm") {
        source = "var module;\n" + source + "\nexport default initSqlJs;";
      }
      const resolver = await getDependencyResolver(root, path, source);
      await writeFile(outputPath, rewriteNpmImports(source, resolver), "utf-8");
    } else {
      await writeFile(outputPath, Buffer.from(await response.arrayBuffer()));
    }
    return outputPath;
  })();
  promise.catch(console.error).then(() => npmRequests.delete(outputPath));
  npmRequests.set(outputPath, promise);
  return promise;
}
async function getDependencyResolver(root, path, input) {
  const body = parseProgram(input);
  const dependencies = /* @__PURE__ */ new Set();
  const { name, range } = parseNpmSpecifier(extractNpmSpecifier(path));
  simple(body, {
    ImportDeclaration: findImport,
    ImportExpression: findImport,
    ExportAllDeclaration: findImport,
    ExportNamedDeclaration: findImport,
    CallExpression: findImportMetaResolve
  });
  function findImport(node) {
    if (node.source && isStringLiteral(node.source)) {
      findImportSource(node.source);
    }
  }
  function findImportMetaResolve(node) {
    if (isImportMetaResolve(node) && isStringLiteral(node.arguments[0])) {
      findImportSource(node.arguments[0]);
    }
  }
  function findImportSource(source) {
    const value = getStringLiteralValue(source);
    if (value.startsWith("/npm/")) {
      const { name: depName, range: depRange } = parseNpmSpecifier(value.slice("/npm/".length));
      if (depName === name)
        return;
      if (depRange && existsSync(join(root, ".observablehq", "cache", "_npm", `${depName}@${depRange}`)))
        return;
      dependencies.add(value);
    }
  }
  const resolutions = /* @__PURE__ */ new Map();
  if (dependencies.size > 0) {
    const pkgPath = await populateNpmCache(root, `/_npm/${name}@${range}/package.json`);
    const pkg = JSON.parse(await readFile(pkgPath, "utf-8"));
    for (const dependency of dependencies) {
      const { name: depName, path: depPath = "+esm" } = parseNpmSpecifier(dependency.slice("/npm/".length));
      const range2 = (name === "arquero" || name === "@uwdata/mosaic-core" || name === "@duckdb/duckdb-wasm") && depName === "apache-arrow" ? "latest" : name === "@uwdata/mosaic-core" && depName === "@duckdb/duckdb-wasm" ? DUCKDB_WASM_VERSION : pkg.dependencies?.[depName] ?? pkg.devDependencies?.[depName] ?? pkg.peerDependencies?.[depName] ?? void console.warn(yellow(`${depName} is an undeclared dependency of ${name}; resolving latest version`));
      resolutions.set(dependency, await resolveNpmImport(root, `${depName}${range2 ? `@${range2}` : ""}/${depPath}`));
    }
  }
  return (specifier) => {
    if (!specifier.startsWith("/npm/"))
      return specifier;
    if (resolutions.has(specifier))
      specifier = resolutions.get(specifier);
    else
      specifier = fromJsDelivrPath(specifier);
    return relativePath(path, specifier);
  };
}
async function initializeNpmVersionCache(root, dir = "_npm") {
  const cache = /* @__PURE__ */ new Map();
  const cacheDir = join(root, ".observablehq", "cache", dir);
  try {
    for (const entry of await readdir(cacheDir)) {
      if (entry.startsWith("@")) {
        for (const subentry of await readdir(join(cacheDir, entry))) {
          const { name, range } = parseNpmSpecifier(`${entry}/${subentry}`);
          const versions = cache.get(name);
          if (versions)
            versions.push(range);
          else
            cache.set(name, [range]);
        }
      } else {
        const { name, range } = parseNpmSpecifier(entry);
        const versions = cache.get(name);
        if (versions)
          versions.push(range);
        else
          cache.set(name, [range]);
      }
    }
  } catch (error) {
    if (!isEnoent(error))
      throw error;
  }
  for (const [key, value] of cache) {
    cache.set(key, rsort(value));
  }
  return cache;
}
const npmVersionCaches = /* @__PURE__ */ new Map();
const npmVersionRequests = /* @__PURE__ */ new Map();
function getNpmVersionCache(root) {
  let cache = npmVersionCaches.get(root);
  if (!cache)
    npmVersionCaches.set(root, cache = initializeNpmVersionCache(root, "_npm"));
  return cache;
}
async function resolveNpmVersion(root, { name, range }) {
  if (range && /^\d+\.\d+\.\d+([-+].*)?$/.test(range))
    return range;
  const cache = await getNpmVersionCache(root);
  const versions = cache.get(name);
  if (versions) {
    for (const version of versions)
      if (!range || satisfies(version, range))
        return version;
  }
  if (range === void 0)
    range = "latest";
  const disttag = validRange(range) ? null : range;
  const href = `https://registry.npmjs.org/${name}${disttag ? `/${disttag}` : ""}`;
  let promise = npmVersionRequests.get(href);
  if (promise)
    return promise;
  promise = async function() {
    const input = formatNpmSpecifier({ name, range });
    process.stdout.write(`npm:${input} ${faint("\u2192")} `);
    const response = await fetch(href, { ...!disttag && { headers: { Accept: "application/vnd.npm.install-v1+json" } } });
    if (!response.ok)
      throw new Error(`unable to fetch: ${href}`);
    const body = await response.json();
    const version = disttag ? body.version : maxSatisfying(Object.keys(body.versions), range);
    if (!version)
      throw new Error(`unable to resolve version: ${input}`);
    const output = formatNpmSpecifier({ name, range: version });
    process.stdout.write(`npm:${output}
`);
    cache.set(name, versions ? rsort(versions.concat(version)) : [version]);
    mkdir(join(root, ".observablehq", "cache", "_npm", output), { recursive: true });
    return version;
  }();
  promise.catch(console.error).then(() => npmVersionRequests.delete(href));
  npmVersionRequests.set(href, promise);
  return promise;
}
async function resolveNpmImport(root, specifier) {
  const {
    name,
    range = name === "@duckdb/duckdb-wasm" ? DUCKDB_WASM_VERSION : void 0,
    path = name === "mermaid" ? "dist/mermaid.esm.min.mjs/+esm" : name === "echarts" ? "dist/echarts.esm.min.js/+esm" : name === "jquery-ui" ? "dist/jquery-ui.js/+esm" : name === "deck.gl" ? "dist.min.js/+esm" : name === "react-dom" ? "client" : "+esm"
  } = parseNpmSpecifier(specifier);
  const version = await resolveNpmVersion(root, { name, range });
  return `/_npm/${name}@${version}/${extname(path) || // npm:foo/bar.js or npm:foo/bar.css
  path === "" || // npm:foo/
  path.endsWith("/") ? path : path === "+esm" ? "_esm.js" : path.replace(/(?:\/\+esm)?$/, "._esm.js")}`;
}
async function resolveNpmImports(root, path) {
  if (!path.startsWith("/_npm/"))
    throw new Error(`invalid npm path: ${path}`);
  await populateNpmCache(root, path);
  return parseImports(join(root, ".observablehq", "cache"), path);
}
function extractNpmSpecifier(path) {
  if (!path.startsWith("/_npm/"))
    throw new Error(`invalid npm path: ${path}`);
  const parts = path.split("/");
  const i = parts[2].startsWith("@") ? 4 : 3;
  const namever = parts.slice(2, i).join("/");
  const subpath = parts.slice(i).join("/");
  return `${namever}/${subpath === "_esm.js" ? "+esm" : subpath.replace(/\._esm\.js$/, "/+esm")}`;
}
function fromJsDelivrPath(path) {
  if (!path.startsWith("/npm/"))
    throw new Error(`invalid jsDelivr path: ${path}`);
  const parts = path.split("/");
  const i = parts[2].startsWith("@") ? 4 : 3;
  const namever = parts.slice(2, i).join("/");
  const subpath = parts.slice(i).join("/");
  return `/_npm/${namever}/${subpath === "+esm" ? "_esm.js" : subpath.replace(/\/\+esm$/, "._esm.js")}`;
}
export {
  extractNpmSpecifier,
  formatNpmSpecifier,
  fromJsDelivrPath,
  getDependencyResolver,
  initializeNpmVersionCache,
  parseNpmSpecifier,
  populateNpmCache,
  resolveNpmImport,
  resolveNpmImports,
  rewriteNpmImports
};
