import fs from "node:fs/promises";
import os from "node:os";
import op from "node:path";
import { CliError, isEnoent } from "./error.js";
const defaultEffects = {
  readFile: (path, encoding) => fs.readFile(path, encoding),
  writeFile: fs.writeFile,
  mkdir: async (path, options) => {
    await fs.mkdir(path, options);
  },
  env: process.env,
  cwd: process.cwd,
  homedir: os.homedir
};
const userConfigName = ".observablehq";
async function getObservableApiKey(effects = defaultEffects) {
  const envVar = "OBSERVABLE_TOKEN";
  if (effects.env[envVar]) {
    return { source: "env", envVar, key: effects.env[envVar] };
  }
  const { config, configPath } = await loadUserConfig();
  if (config.auth?.key) {
    return { source: "file", filePath: configPath, key: config.auth.key };
  }
  return null;
}
async function setObservableApiKey(info) {
  const { config, configPath } = await loadUserConfig();
  if (info) {
    config.auth = info;
  } else {
    delete config.auth;
  }
  await writeUserConfig({ config, configPath });
}
function getDefaultDeployConfigPath(sourceRoot, effects) {
  return op.join(effects.cwd(), sourceRoot, ".observablehq", "deploy.json");
}
async function getDeployConfig(sourceRoot, deployConfigPath, effects = defaultEffects) {
  if (deployConfigPath === void 0)
    deployConfigPath = getDefaultDeployConfigPath(sourceRoot, effects);
  let config = null;
  try {
    const content = await effects.readFile(deployConfigPath, "utf8");
    config = JSON.parse(content);
  } catch (error) {
    if (!isEnoent(error)) {
      const message = error instanceof Error ? error.message : String(error);
      throw new CliError(`Could not read config file at ${deployConfigPath}: ${message}`, { cause: error });
    }
  }
  let { projectId, projectSlug, workspaceLogin } = config ?? {};
  if (typeof projectId !== "string")
    projectId = null;
  if (typeof projectSlug !== "string")
    projectSlug = null;
  if (typeof workspaceLogin !== "string")
    workspaceLogin = null;
  return { projectId, projectSlug, workspaceLogin };
}
async function setDeployConfig(sourceRoot, deployConfigPath, newConfig, effects = defaultEffects) {
  if (deployConfigPath === void 0)
    deployConfigPath = getDefaultDeployConfigPath(sourceRoot, effects);
  const oldConfig = await getDeployConfig(sourceRoot, deployConfigPath);
  const merged = { ...oldConfig, ...newConfig };
  await effects.mkdir(op.dirname(deployConfigPath), { recursive: true });
  await effects.writeFile(deployConfigPath, JSON.stringify(merged, null, 2) + "\n");
}
async function loadUserConfig(effects = defaultEffects) {
  const homeConfigPath = op.join(effects.homedir(), userConfigName);
  function* pathsToTry() {
    let cursor = op.resolve(effects.cwd());
    while (true) {
      yield op.join(cursor, userConfigName);
      const nextCursor = op.dirname(cursor);
      if (nextCursor === cursor)
        break;
      cursor = nextCursor;
    }
    yield homeConfigPath;
  }
  for (const configPath of pathsToTry()) {
    let content = null;
    try {
      content = await effects.readFile(configPath, "utf8");
    } catch (error) {
      if (!isEnoent(error))
        throw error;
    }
    if (content !== null) {
      try {
        return { config: JSON.parse(content), configPath };
      } catch (error) {
        throw new CliError(`Problem parsing config file at ${configPath}: ${error}`, { cause: error });
      }
    }
  }
  return { config: {}, configPath: homeConfigPath };
}
async function writeUserConfig({ configPath, config }, effects = defaultEffects) {
  await effects.writeFile(configPath, JSON.stringify(config, null, 2));
}
export {
  defaultEffects,
  getDeployConfig,
  getObservableApiKey,
  loadUserConfig,
  setDeployConfig,
  setObservableApiKey
};
