import he from "he";
import MiniSearch from "minisearch";
import { findModule } from "./javascript/module.js";
import { faint, strikethrough } from "./tty.js";
const indexCache = /* @__PURE__ */ new WeakMap();
const reindexDelay = 10 * 60 * 1e3;
const defaultEffects = {
  logger: console,
  output: process.stdout
};
const indexOptions = {
  fields: ["title", "text", "keywords"],
  // fields to return with search results
  storeFields: ["title"],
  processTerm(term) {
    return (term.match(/\p{N}/gu)?.length ?? 0) > 6 ? null : term.slice(0, 15).toLowerCase();
  }
};
async function searchIndex(config, paths = getDefaultSearchPaths(config), effects = defaultEffects) {
  const { pages, search, normalizePath } = config;
  if (!search)
    return "{}";
  const cached = indexCache.get(pages);
  if (cached && cached.freshUntil > Date.now())
    return cached.json;
  const index = new MiniSearch(indexOptions);
  for await (const result of indexPages(config, paths, effects))
    index.add(normalizeResult(result, normalizePath));
  if (search.index)
    for await (const result of search.index())
      index.add(normalizeResult(result, normalizePath));
  const json = JSON.stringify(
    Object.assign(
      {
        options: {
          fields: indexOptions.fields,
          storeFields: indexOptions.storeFields
        }
      },
      index.toJSON()
    )
  );
  indexCache.set(pages, { json, freshUntil: Date.now() + reindexDelay });
  return json;
}
async function* indexPages(config, paths, effects) {
  const { pages, loaders } = config;
  const pagePaths = /* @__PURE__ */ new Set(["/index"]);
  for (const p of pages) {
    if (p.path !== null)
      pagePaths.add(p.path);
    if ("pages" in p)
      for (const { path } of p.pages)
        pagePaths.add(path);
  }
  for await (const path of paths) {
    const { body, title, data } = await loaders.loadPage(path, { ...config, path });
    const listed = pagePaths.has(path);
    const indexed = data?.index === void 0 ? listed : Boolean(data.index);
    if (!indexed) {
      if (listed)
        effects.logger.log(`${faint("index")} ${strikethrough(path)} ${faint("(skipped)")}`);
      continue;
    }
    const text = he.decode(
      body.replaceAll(/[\n\r]/g, " ").replaceAll(/<style\b.*<\/style\b[^>]*>/gi, " ").replaceAll(/<[^>]+>/g, " ")
    ).normalize("NFD").replaceAll(/[\u0300-\u036f]/g, "").replace(/[^\p{L}\p{N}]/gu, " ");
    effects.logger.log(`${faint("index")} ${path}`);
    yield { path, title, text, keywords: normalizeKeywords(data?.keywords) };
  }
}
async function* getDefaultSearchPaths(config) {
  const { root, loaders } = config;
  for await (const path of config.paths()) {
    if (path.endsWith(".js") && findModule(root, path))
      continue;
    if (loaders.find(path))
      continue;
    yield path;
  }
}
function normalizeResult({ path, keywords, ...rest }, normalizePath) {
  return { id: normalizePath(path), keywords: keywords ?? "", ...rest };
}
function normalizeKeywords(keywords) {
  return keywords ? String(keywords) : "";
}
export {
  searchIndex
};
