import os from "node:os";
const DEFAULT_CONCURRENCY = Math.max(Math.min(os.cpus().length * 2, 8), 2);
async function runAllWithConcurrencyLimit(tasks, worker, { maxConcurrency = DEFAULT_CONCURRENCY } = {}) {
  const queue = Array.from(tasks);
  const pending = /* @__PURE__ */ new Set();
  let index = 0;
  while (queue.length) {
    if (pending.size >= maxConcurrency) {
      await Promise.race(pending);
      continue;
    }
    const item = queue.shift();
    if (!item)
      throw new Error("unexpectedly out of items");
    const promise = worker(item, index++);
    pending.add(promise);
    promise.finally(() => pending.delete(promise));
  }
  await Promise.all(pending);
}
class RateLimiter {
  constructor(ratePerSecond) {
    this.ratePerSecond = ratePerSecond;
    this._nextTick = Promise.resolve();
  }
  // This works by chaining together promises, one for each call to `this.wait`.
  // This implicitly forms a queue of callers. The important thing is that we
  // never have two of this function's `setTimeout`s running concurrently, since
  // that could cause us to exceed the rate limit.
  _nextTick;
  /** Wait long enough to avoid going over the rate limit. */
  async wait() {
    const nextTick = this._nextTick;
    this._nextTick = nextTick.then(() => new Promise((res) => setTimeout(res, 1e3 / this.ratePerSecond)));
    await nextTick;
  }
}
export {
  RateLimiter,
  runAllWithConcurrencyLimit
};
