import { existsSync } from "node:fs";
import { utimes, writeFile } from "node:fs/promises";
import { join } from "node:path/posix";
import wrapAnsi from "wrap-ansi";
import { CliError } from "./error.js";
import { prepareOutput } from "./files.js";
import { getObservableUiOrigin } from "./observableApiClient.js";
import { bold, cyan, defaultEffects as defaultTtyEffects, faint, inverse, link, reset } from "./tty.js";
const defaultEffects = {
  ...defaultTtyEffects,
  async prepareOutput(outputPath) {
    await prepareOutput(outputPath);
  },
  existsSync(outputPath) {
    return existsSync(outputPath);
  },
  async writeFile(outputPath, contents) {
    await writeFile(outputPath, contents);
  },
  async touch(outputPath, date) {
    await utimes(outputPath, date = new Date(date), date);
  }
};
async function convert(inputs, { output, force = false, files: includeFiles = true }, effects = defaultEffects) {
  const { clack } = effects;
  clack.intro(`${inverse(" observable convert ")} ${faint(`v${"1.13.3"}`)}`);
  let n = 0;
  for (const input of inputs) {
    let start = Date.now();
    let s = clack.spinner();
    const url = resolveInput(input);
    const name = inferFileName(url);
    const path = join(output, name);
    if (await maybeFetch(path, force, effects)) {
      s.start(`Downloading ${bold(path)}`);
      const response = await fetch(url);
      if (!response.ok)
        throw new Error(`error fetching ${url}: ${response.status}`);
      const { nodes, files, update_time } = await response.json();
      s.stop(`Downloaded ${bold(path)} ${faint(`in ${(Date.now() - start).toLocaleString("en-US")}ms`)}`);
      await effects.prepareOutput(path);
      await effects.writeFile(path, convertNodes(nodes));
      await effects.touch(path, update_time);
      n++;
      if (includeFiles) {
        for (const file of files) {
          const path2 = join(output, file.name);
          if (await maybeFetch(path2, force, effects)) {
            start = Date.now();
            s = clack.spinner();
            s.start(`Downloading ${bold(path2)}`);
            const response2 = await fetch(file.download_url);
            if (!response2.ok)
              throw new Error(`error fetching ${file.download_url}: ${response2.status}`);
            const buffer = Buffer.from(await response2.arrayBuffer());
            s.stop(`Downloaded ${bold(path2)} ${faint(`in ${(Date.now() - start).toLocaleString("en-US")}ms`)}`);
            await effects.prepareOutput(path2);
            await effects.writeFile(path2, buffer);
            await effects.touch(path2, file.create_time);
            n++;
          }
        }
      }
    }
  }
  clack.note(
    wrapAnsi(
      "Due to syntax differences between Observable notebooks and Observable Framework, converted notebooks may require further changes to function correctly. To learn more about JavaScript in Framework, please read:\n\n" + reset(cyan(link("https://observablehq.com/framework/javascript"))),
      Math.min(64, effects.outputColumns)
    ),
    "Note"
  );
  clack.outro(
    `${inputs.length} notebook${inputs.length === 1 ? "" : "s"} converted; ${n} file${n === 1 ? "" : "s"} written`
  );
}
async function maybeFetch(path, force, effects) {
  const { clack } = effects;
  if (effects.existsSync(path) && !force) {
    const choice = await clack.confirm({ message: `${bold(path)} already exists; replace?`, initialValue: false });
    if (!choice)
      return false;
    if (clack.isCancel(choice))
      throw new CliError("Stopped convert", { print: false });
  }
  return true;
}
function convertNodes(nodes) {
  let string = "";
  let first = true;
  for (const node of nodes) {
    if (first)
      first = false;
    else
      string += "\n";
    string += convertNode(node);
  }
  return string;
}
function convertNode(node) {
  let string = "";
  if (node.mode !== "md")
    string += `\`\`\`${node.mode}${node.pinned ? " echo" : ""}
`;
  string += `${node.value}
`;
  if (node.mode !== "md")
    string += "```\n";
  return string;
}
function inferFileName(input) {
  return new URL(input).pathname.replace(/^\/document(\/@[^/]+)?\//, "").replace(/\//g, ",") + ".md";
}
function resolveInput(input) {
  let url;
  if (isIdSpecifier(input))
    url = new URL(`/d/${input}`, getObservableUiOrigin());
  else if (isSlugSpecifier(input))
    url = new URL(`/${input}`, getObservableUiOrigin());
  else
    url = new URL(input);
  url.host = `api.${url.host}`;
  url.pathname = `/document${url.pathname.replace(/^\/d\//, "/")}`;
  return String(url);
}
function isIdSpecifier(string) {
  return /^([0-9a-f]{16})(?:@(\d+)|~(\d+)|@(\w+))?$/.test(string);
}
function isSlugSpecifier(string) {
  return /^(?:@([0-9a-z_-]+))\/([0-9a-z_-]+(?:\/[0-9]+)?)(?:@(\d+)|~(\d+)|@(\w+))?$/.test(string);
}
export {
  convert,
  convertNode,
  convertNodes,
  inferFileName,
  resolveInput
};
