import he from "he";
import hljs from "highlight.js";
import { JSDOM, VirtualConsole } from "jsdom";
import { isAssetPath, parseRelativeUrl, relativePath, resolveLocalPath, resolvePath } from "./path.js";
const ASSET_ATTRIBUTES = [
  ["a[href][download]", "href"],
  ["audio source[src]", "src"],
  ["audio[src]", "src"],
  ["img[src]", "src"],
  ["img[srcset]", "srcset"],
  ["link[href]", "href"],
  ["picture source[srcset]", "srcset"],
  ["video source[src]", "src"],
  ["video[src]", "src"]
];
const PATH_ATTRIBUTES = [
  ["a[href]", "href"],
  ["audio source[src]", "src"],
  ["audio[src]", "src"],
  ["img[src]", "src"],
  ["img[srcset]", "srcset"],
  ["link[href]", "href"],
  ["picture source[srcset]", "srcset"],
  ["video source[src]", "src"],
  ["video[src]", "src"]
];
function isJavaScript({ type }) {
  if (!type)
    return true;
  type = type.toLowerCase();
  return type === "text/javascript" || type === "application/javascript" || type === "module";
}
function parseHtml(html2) {
  return new JSDOM(`<!DOCTYPE html><body>${html2}`, { virtualConsole: new VirtualConsole() }).window;
}
function findAssets(html2, path) {
  const { document } = parseHtml(html2);
  const files = /* @__PURE__ */ new Set();
  const anchors = /* @__PURE__ */ new Set();
  const localLinks = /* @__PURE__ */ new Set();
  const localImports = /* @__PURE__ */ new Set();
  const globalImports = /* @__PURE__ */ new Set();
  const staticImports = /* @__PURE__ */ new Set();
  const maybeFile = (specifier) => {
    if (isAssetPath(specifier)) {
      const localPath = resolveLocalPath(path, specifier);
      if (!localPath)
        return console.warn(`non-local asset path: ${specifier}`);
      files.add(relativePath(path, localPath));
    } else {
      globalImports.add(specifier);
    }
  };
  for (const [selector, src] of ASSET_ATTRIBUTES) {
    for (const element of document.querySelectorAll(selector)) {
      if (isExternal(element))
        continue;
      const source = decodeURI(element.getAttribute(src));
      if (src === "srcset") {
        for (const s of parseSrcset(source)) {
          maybeFile(s);
        }
      } else {
        maybeFile(source);
      }
    }
  }
  for (const script of document.querySelectorAll("script[src]")) {
    if (isExternal(script))
      continue;
    let src = script.getAttribute("src");
    if (isJavaScript(script)) {
      if (isAssetPath(src)) {
        const localPath = resolveLocalPath(path, src);
        if (!localPath) {
          console.warn(`non-local asset path: ${src}`);
          continue;
        }
        localImports.add(src = relativePath(path, localPath));
      } else {
        globalImports.add(src);
      }
      if (script.getAttribute("type")?.toLowerCase() === "module" && !script.hasAttribute("async")) {
        staticImports.add(src);
      }
    } else {
      maybeFile(src);
    }
  }
  for (const element of document.querySelectorAll("[id],[name]")) {
    if (isExternal(element))
      continue;
    anchors.add(element.getAttribute("id") ?? element.getAttribute("name"));
  }
  for (const a of document.querySelectorAll("a[href]")) {
    if (isExternal(a) || a.hasAttribute("download"))
      continue;
    const href = a.getAttribute("href");
    if (/^\w+:/.test(href))
      continue;
    const { pathname, search, hash } = parseRelativeUrl(href);
    localLinks.add(resolvePath(path, pathname).replace(/\.html$/i, "").replace(/\/$/, "/index") + search + hash);
  }
  return { files, localImports, globalImports, staticImports, localLinks, anchors };
}
function rewriteHtmlPaths(html2, path) {
  const { document } = parseHtml(html2);
  const resolvePath2 = (specifier) => {
    return isAssetPath(specifier) ? relativePath(path, specifier) : specifier;
  };
  for (const [selector, src] of PATH_ATTRIBUTES) {
    for (const element of document.querySelectorAll(selector)) {
      if (isExternal(element))
        continue;
      const source = decodeURI(element.getAttribute(src));
      element.setAttribute(src, src === "srcset" ? resolveSrcset(source, resolvePath2) : encodeURI(resolvePath2(source)));
    }
  }
  return document.body.innerHTML;
}
function rewriteHtml(html2, { resolveFile = String, resolveImport = String, resolveScript = String, resolveLink = String }) {
  const { document } = parseHtml(html2);
  const resolvePath2 = (specifier) => {
    return isAssetPath(specifier) ? resolveFile(specifier) : resolveImport(specifier);
  };
  for (const [selector, src] of ASSET_ATTRIBUTES) {
    for (const element of document.querySelectorAll(selector)) {
      if (isExternal(element))
        continue;
      const source = decodeURI(element.getAttribute(src));
      element.setAttribute(src, src === "srcset" ? resolveSrcset(source, resolvePath2) : encodeURI(resolvePath2(source)));
    }
  }
  for (const script of document.querySelectorAll("script[src]")) {
    if (isExternal(script))
      continue;
    const src = decodeURI(script.getAttribute("src"));
    script.setAttribute("src", encodeURI((isJavaScript(script) ? resolveScript : resolveFile)(src)));
  }
  for (const a of document.querySelectorAll("a[href]")) {
    if (isExternal(a))
      continue;
    const href = decodeURI(a.getAttribute("href"));
    a.setAttribute("href", encodeURI(resolveLink(href)));
    if (!/^(\w+:)/.test(href))
      continue;
    if (!a.hasAttribute("target"))
      a.setAttribute("target", "_blank");
    if (!a.hasAttribute("rel"))
      a.setAttribute("rel", "noopener noreferrer");
  }
  for (const code of document.querySelectorAll("code[class*='language-']")) {
    const language = [...code.classList].find((c) => c.startsWith("language-"))?.slice("language-".length);
    if (!language)
      continue;
    if (code.parentElement?.tagName === "PRE")
      code.parentElement.setAttribute("data-language", language);
    if (!hljs.getLanguage(language))
      continue;
    let html3 = "";
    code.normalize();
    for (const child of code.childNodes) {
      html3 += isText(child) ? hljs.highlight(child.textContent, { language }).value : isElement(child) ? child.outerHTML : isComment(child) ? `<!--${he.escape(child.data)}-->` : "";
    }
    code.innerHTML = html3;
  }
  for (const h of document.querySelectorAll("h1[id], h2[id], h3[id], h4[id]")) {
    const a = document.createElement("a");
    a.className = "observablehq-header-anchor";
    a.href = `#${h.id}`;
    a.append(...h.childNodes);
    h.append(a);
  }
  for (let child = document.body.firstChild; child; child = child.nextSibling) {
    if (isRoot(child)) {
      const parent = document.createElement("span");
      const loading = findLoading(child);
      child.replaceWith(parent);
      if (loading)
        parent.appendChild(loading);
      parent.appendChild(child);
      child = parent;
    }
  }
  for (const l of document.querySelectorAll("observablehq-loading")) {
    if (!l.nextSibling || !isRoot(l.nextSibling) || l.namespaceURI !== "http://www.w3.org/1999/xhtml") {
      l.remove();
    }
  }
  return document.body.innerHTML;
}
function parseSrcset(srcset) {
  return srcset.trim().split(/\s*,\s*/).filter((src) => src).map((src) => src.split(/\s+/)[0]);
}
function resolveSrcset(srcset, resolve) {
  return srcset.trim().split(/\s*,\s*/).filter((src) => src).map((src) => {
    const parts = src.split(/\s+/);
    const path = resolve(parts[0]);
    if (path)
      parts[0] = encodeURI(path);
    return parts.join(" ");
  }).join(", ");
}
function isText(node) {
  return node.nodeType === 3;
}
function isComment(node) {
  return node.nodeType === 8;
}
function isElement(node) {
  return node.nodeType === 1;
}
function isRoot(node) {
  return isComment(node) && /^:[0-9a-f]{8}(?:-\d+)?:$/.test(node.data);
}
function isLoading(node) {
  return isElement(node) && node.tagName === "OBSERVABLEHQ-LOADING";
}
function isExternal(a) {
  return /(?:^|\s)external(?:\s|$)/i.test(a.getAttribute("rel") ?? "");
}
function findLoading(node) {
  const sibling = node.previousSibling;
  return sibling && isLoading(sibling) ? sibling : null;
}
class Html {
  constructor(html2) {
    this.html = html2;
  }
  static unsafe(html2) {
    return new Html(html2);
  }
  toString() {
    return this.html;
  }
}
function html(strings, ...values) {
  const parts = [];
  for (let i = 0; i < strings.length; ++i) {
    parts.push(strings[i]);
    if (i < values.length) {
      const value = values[i];
      if (value == null)
        continue;
      if (typeof value[Symbol.iterator] === "function") {
        for (const v of value) {
          if (v == null)
            continue;
          parts.push(v instanceof Html ? v.html : he.escape(String(v)));
        }
      } else {
        parts.push(value instanceof Html ? value.html : he.escape(String(value)));
      }
    }
  }
  return Html.unsafe(parts.join(""));
}
html.unsafe = Html.unsafe;
export {
  Html,
  findAssets,
  html,
  isComment,
  isElement,
  isJavaScript,
  isText,
  parseHtml,
  rewriteHtml,
  rewriteHtmlPaths
};
