import { existsSync } from "node:fs";
import { copyFile, readFile, writeFile } from "node:fs/promises";
import { createRequire } from "node:module";
import op from "node:path";
import { extname, join } from "node:path/posix";
import { pathToFileURL } from "node:url";
import commonjs from "@rollup/plugin-commonjs";
import json from "@rollup/plugin-json";
import { nodeResolve } from "@rollup/plugin-node-resolve";
import virtual from "@rollup/plugin-virtual";
import { packageDirectory } from "pkg-dir";
import { rollup } from "rollup";
import esbuild from "rollup-plugin-esbuild";
import { prepareOutput, toOsPath } from "./files.js";
import { annotatePath } from "./javascript/annotate.js";
import { isJavaScript, parseImports } from "./javascript/imports.js";
import { parseNpmSpecifier, rewriteNpmImports } from "./npm.js";
import { isPathImport, relativePath } from "./path.js";
import { faint } from "./tty.js";
async function resolveNodeImport(root, spec) {
  return resolveNodeImportInternal(op.join(root, ".observablehq", "cache", "_node"), root, spec);
}
const bundlePromises = /* @__PURE__ */ new Map();
async function resolveNodeImportInternal(cacheRoot, packageRoot, spec) {
  const { name, path = "." } = parseNpmSpecifier(spec);
  const require2 = createRequire(pathToFileURL(op.join(packageRoot, "/")));
  const pathResolution = require2.resolve(spec);
  const packageResolution = await packageDirectory({ cwd: op.dirname(pathResolution) });
  if (!packageResolution)
    throw new Error(`unable to resolve package.json: ${spec}`);
  const { version } = JSON.parse(await readFile(op.join(packageResolution, "package.json"), "utf-8"));
  const resolution = `${name}@${version}/${extname(path) ? path : path === "." ? "index.js" : `${path}.js`}`;
  const outputPath = op.join(cacheRoot, toOsPath(resolution));
  const resolutionPath = `/_node/${resolution}`;
  if (existsSync(outputPath))
    return resolutionPath;
  let promise = bundlePromises.get(outputPath);
  if (promise)
    return promise;
  promise = (async () => {
    console.log(`${spec} ${faint("\u2192")} ${outputPath}`);
    await prepareOutput(outputPath);
    if (isJavaScript(pathResolution)) {
      await writeFile(outputPath, await bundle(resolutionPath, spec, require2, cacheRoot, packageResolution), "utf-8");
    } else {
      await copyFile(pathResolution, outputPath);
    }
    return resolutionPath;
  })();
  promise.catch(console.error).then(() => bundlePromises.delete(outputPath));
  bundlePromises.set(outputPath, promise);
  return promise;
}
async function resolveNodeImports(root, path) {
  if (!path.startsWith("/_node/"))
    throw new Error(`invalid node path: ${path}`);
  return parseImports(join(root, ".observablehq", "cache"), path);
}
function extractNodeSpecifier(path) {
  if (!path.startsWith("/_node/"))
    throw new Error(`invalid node path: ${path}`);
  return path.replace(/^\/_node\//, "");
}
function isBadCommonJs(specifier) {
  const { name } = parseNpmSpecifier(specifier);
  return name === "react" || name === "react-dom" || name === "react-is" || name === "scheduler";
}
function shimCommonJs(specifier, require2) {
  return `export {${Object.keys(require2(specifier))}} from ${annotatePath(specifier)};
`;
}
async function bundle(path, input, require2, cacheRoot, packageRoot) {
  const bundle2 = await rollup({
    input: isBadCommonJs(input) ? "-" : input,
    plugins: [
      ...isBadCommonJs(input) ? [virtual({ "-": shimCommonJs(input, require2) })] : [],
      importResolve(input, cacheRoot, packageRoot),
      nodeResolve({ browser: true, rootDir: packageRoot }),
      json(),
      commonjs({
        esmExternals: true,
        requireReturnsDefault: "preferred"
      }),
      esbuild({
        format: "esm",
        platform: "browser",
        target: ["es2022", "chrome96", "firefox96", "safari16", "node18"],
        exclude: [],
        // don’t exclude node_modules
        define: { "process.env.NODE_ENV": JSON.stringify("production") },
        minify: true
      })
    ],
    external(source) {
      return source.startsWith("/_node/");
    },
    onwarn(message, warn) {
      if (message.code === "CIRCULAR_DEPENDENCY")
        return;
      warn(message);
    }
  });
  try {
    const output = await bundle2.generate({ format: "es", exports: "named" });
    const code = output.output.find((o) => o.type === "chunk").code;
    return rewriteNpmImports(code, (i) => i.startsWith("/_node/") ? relativePath(path, i) : i);
  } finally {
    await bundle2.close();
  }
}
function importResolve(input, cacheRoot, packageRoot) {
  async function resolve(specifier) {
    return typeof specifier !== "string" || // AST node?
    isNodeBuiltin(specifier) || // node built-in, e.g., "node:fs" or "fs"
    isPathImport(specifier) || // relative path, e.g., ./foo.js
    /^\0?[\w-]+:/.test(specifier) || // windows file path, https: URL, \x00node-resolve:, etc.
    specifier === input ? null : { id: await resolveNodeImportInternal(cacheRoot, packageRoot, specifier), external: true };
  }
  return {
    name: "resolve-import",
    resolveId: resolve,
    resolveDynamicImport: resolve
  };
}
function isNodeBuiltin(specifier) {
  return specifier.startsWith("node:") || nodeBuiltins.has(specifier.replace(/\/.*/, ""));
}
const nodeBuiltins = /* @__PURE__ */ new Set([
  "_http_agent",
  "_http_client",
  "_http_common",
  "_http_incoming",
  "_http_outgoing",
  "_http_server",
  "_stream_duplex",
  "_stream_passthrough",
  "_stream_readable",
  "_stream_transform",
  "_stream_wrap",
  "_stream_writable",
  "_tls_common",
  "_tls_wrap",
  "assert",
  "async_hooks",
  "buffer",
  "child_process",
  "cluster",
  "console",
  "constants",
  "crypto",
  "dgram",
  "diagnostics_channel",
  "dns",
  "domain",
  "events",
  "fs",
  "http",
  "http2",
  "https",
  "inspector",
  "module",
  "net",
  "os",
  "path",
  "perf_hooks",
  "process",
  "punycode",
  "querystring",
  "readline",
  "repl",
  "stream",
  "string_decoder",
  "sys",
  "timers",
  "tls",
  "trace_events",
  "tty",
  "url",
  "util",
  "v8",
  "vm",
  "wasi",
  "worker_threads",
  "zlib"
]);
export {
  extractNodeSpecifier,
  isNodeBuiltin,
  resolveNodeImport,
  resolveNodeImports
};
