import fs from "node:fs/promises";
import { CliError, HttpError, isApiError } from "./error.js";
import { formatByteSize } from "./format.js";
import { faint, red } from "./tty.js";
const MIN_RATE_LIMIT_RETRY_AFTER = 1;
function getObservableUiOrigin(env = process.env) {
  const urlText = env["OBSERVABLE_ORIGIN"] ?? "https://observablehq.com";
  try {
    return new URL(urlText);
  } catch (error) {
    throw new CliError(`Invalid OBSERVABLE_ORIGIN: ${urlText}`, { cause: error });
  }
}
function getObservableApiOrigin(env = process.env) {
  const urlText = env["OBSERVABLE_API_ORIGIN"];
  if (urlText) {
    try {
      return new URL(urlText);
    } catch (error) {
      throw new CliError(`Invalid OBSERVABLE_API_ORIGIN: ${urlText}`, { cause: error });
    }
  }
  const uiOrigin = getObservableUiOrigin(env);
  uiOrigin.hostname = "api." + uiOrigin.hostname;
  return uiOrigin;
}
class ObservableApiClient {
  _apiHeaders;
  _apiOrigin;
  _clack;
  _rateLimit = null;
  constructor({ apiKey, apiOrigin = getObservableApiOrigin(), clack }) {
    this._apiOrigin = apiOrigin;
    this._apiHeaders = {
      Accept: "application/json",
      "User-Agent": `Observable Framework ${"1.13.3"}`,
      "X-Observable-Api-Version": "2023-12-06"
    };
    this._clack = clack;
    if (apiKey)
      this.setApiKey(apiKey);
  }
  setApiKey(apiKey) {
    this._apiHeaders["Authorization"] = `apikey ${apiKey.key}`;
  }
  async _fetch(url, options) {
    let response;
    const doFetch = async () => await fetch(url, { ...options, headers: { ...this._apiHeaders, ...options.headers } });
    try {
      response = await doFetch();
    } catch (error) {
      if (error instanceof Error && error.message === "fetch failed")
        console.error(error);
      throw error;
    }
    if (response.status === 429) {
      if (this._rateLimit === null) {
        let retryAfter = +response.headers.get("Retry-After");
        if (isNaN(retryAfter) || retryAfter < MIN_RATE_LIMIT_RETRY_AFTER)
          retryAfter = MIN_RATE_LIMIT_RETRY_AFTER;
        this._clack.log.warn(`Hit server rate limit. Waiting for ${retryAfter} seconds.`);
        this._rateLimit = new Promise((resolve) => setTimeout(resolve, retryAfter * 1e3));
      }
      await this._rateLimit;
      response = await doFetch();
    }
    if (!response.ok) {
      let details = await response.text();
      try {
        details = JSON.parse(details);
      } catch (error2) {
      }
      const error = new HttpError(
        `Unexpected response status ${JSON.stringify(response.status)} for ${options.method ?? "GET"} ${url.href}`,
        response.status,
        { details }
      );
      if (response.status === 400 && isApiError(error) && error.details.errors.some((e) => e.code === "VERSION_MISMATCH")) {
        console.log(red("The version of Observable Framework you are using is not compatible with the server."));
        console.log(faint(`Expected ${details.errors[0].meta.expected}, but using ${details.errors[0].meta.actual}`));
      }
      throw error;
    }
    if (response.status === 204)
      return null;
    if (response.headers.get("Content-Type")?.startsWith("application/json"))
      return await response.json();
    return await response.text();
  }
  async getCurrentUser() {
    return await this._fetch(new URL("/cli/user", this._apiOrigin), { method: "GET" });
  }
  async getProject({
    workspaceLogin,
    projectSlug
  }) {
    const url = new URL(`/cli/project/@${workspaceLogin}/${projectSlug}`, this._apiOrigin);
    return await this._fetch(url, { method: "GET" });
  }
  async postProject({
    title,
    slug,
    workspaceId,
    accessLevel
  }) {
    return await this._fetch(new URL("/cli/project", this._apiOrigin), {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ title, slug, workspace: workspaceId, accessLevel })
    });
  }
  async getWorkspaceProjects(workspaceLogin) {
    const pages = await this._fetch(
      new URL(`/cli/workspace/@${workspaceLogin}/projects`, this._apiOrigin),
      { method: "GET" }
    );
    return pages.results;
  }
  async getDeploy(deployId) {
    return await this._fetch(new URL(`/cli/deploy/${deployId}`, this._apiOrigin), { method: "GET" });
  }
  async postDeploy({ projectId, message }) {
    const data = await this._fetch(new URL(`/cli/project/${projectId}/deploy`, this._apiOrigin), {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ message })
    });
    return data.id;
  }
  async postDeployFile(deployId, filePath, relativePath) {
    const buffer = await fs.readFile(filePath);
    return await this.postDeployFileContents(deployId, buffer, relativePath);
  }
  async postDeployFileContents(deployId, contents, relativePath) {
    if (typeof contents === "string")
      contents = Buffer.from(contents);
    const url = new URL(`/cli/deploy/${deployId}/file`, this._apiOrigin);
    const body = new FormData();
    const blob = new Blob([contents]);
    body.append("file", blob);
    body.append("client_name", relativePath);
    try {
      await this._fetch(url, { method: "POST", body });
    } catch (error) {
      const message = error instanceof Error ? error.message : `${error}`;
      throw new CliError(`While uploading ${relativePath} (${formatByteSize(contents.length)}): ${message}`, {
        cause: error
      });
    }
  }
  async postDeployManifest(deployId, files) {
    return await this._fetch(new URL(`/cli/deploy/${deployId}/manifest`, this._apiOrigin), {
      method: "POST",
      headers: { "content-type": "application/json" },
      body: JSON.stringify({ files })
    });
  }
  async postDeployUploaded(deployId, buildManifest) {
    return await this._fetch(new URL(`/cli/deploy/${deployId}/uploaded`, this._apiOrigin), {
      method: "POST",
      headers: { "content-type": "application/json" },
      body: JSON.stringify(buildManifest)
    });
  }
  async postAuthRequest(options) {
    return await this._fetch(new URL("/cli/auth/request", this._apiOrigin), {
      method: "POST",
      headers: { "content-type": "application/json" },
      body: JSON.stringify(options)
    });
  }
  async postAuthRequestPoll(id) {
    return await this._fetch(new URL("/cli/auth/request/poll", this._apiOrigin), {
      method: "POST",
      headers: { "content-type": "application/json" },
      body: JSON.stringify({ id })
    });
  }
}
export {
  ObservableApiClient,
  getObservableApiOrigin,
  getObservableUiOrigin
};
