import { ascending, max } from "d3-array";
import { stratify } from "d3-hierarchy";
function tree(items) {
  const lines = [];
  stratify().path(
    ([path]) => path.replace(/\.md$/, "") + // remove .md extension
    (path.endsWith("/") ? "" : "?")
    // distinguish files from folders
  )([...items, ["/"]]).sort(treeOrder).count().eachAfter((node) => {
    const children = node.children;
    if (!children)
      return;
    let height = 0;
    for (let i = 0; i < children.length; ++i) {
      let child = children[i];
      if (child.children?.length === 1) {
        child = child.children[0];
        children[i] = child;
        child.parent = node;
        --child.depth;
      }
      height = Math.max(height, child.height + 1);
    }
    node.height = height;
  }).eachBefore((node) => {
    let p = node;
    let indent = "";
    if (p.parent) {
      indent = (hasFollowingSibling(p) ? "\u251C\u2500\u2500 " : "\u2514\u2500\u2500 ") + indent;
      while ((p = p.parent).parent) {
        indent = (hasFollowingSibling(p) ? "\u2502\xA0\xA0 " : " \xA0\xA0 ") + indent;
      }
    }
    lines.push([
      indent || "\u250C",
      node.id.slice(node.parent?.id === "/" ? 1 : (node.parent?.id.length ?? 0) + 1).replace(/\?$/, ""),
      node.height ? ` (${node.value.toLocaleString("en-US")} page${node.value === 1 ? "" : "s"})` : "",
      node
    ]);
  });
  const width = (max(lines, ([indent, name, description]) => indent.length + description.length + stringLength(name)) || 0) + 1;
  return lines.map(([indent, name, description, node]) => [
    indent,
    name,
    description + " ".repeat(width - stringLength(name) - description.length - indent.length),
    node
  ]);
}
function stringLength(string) {
  return [...new Intl.Segmenter().segment(string)].length;
}
function hasFollowingSibling(node) {
  return node.parent != null && node.parent.children.indexOf(node) < node.parent.children.length - 1;
}
function treeOrder(a, b) {
  return ascending(!a.children, !b.children) || ascending(a.id, b.id);
}
export {
  tree
};
