import { Parser, tokTypes } from "acorn";
import { checkAssignments } from "./assignments.js";
import { findAwaits } from "./awaits.js";
import { findDeclarations } from "./declarations.js";
import { findFiles } from "./files.js";
import { findExports, findImports } from "./imports.js";
import { checkParams } from "./params.js";
import { findReferences } from "./references.js";
import { syntaxError } from "./syntaxError.js";
const acornOptions = {
  ecmaVersion: 13,
  sourceType: "module"
};
function parseJavaScript(input, options) {
  const { inline = false, path, params } = options;
  let expression = maybeParseExpression(input);
  if (expression?.type === "ClassExpression" && expression.id)
    expression = null;
  if (expression?.type === "FunctionExpression" && expression.id)
    expression = null;
  if (!expression && inline)
    throw new SyntaxError("invalid expression");
  const body = expression ?? parseProgram(input);
  const exports = findExports(body);
  if (exports.length)
    throw syntaxError("Unexpected token 'export'", exports[0], input);
  const references = findReferences(body);
  if (params)
    checkParams(body, input, params);
  checkAssignments(body, references, input);
  return {
    body,
    declarations: expression ? null : findDeclarations(body, input),
    references,
    files: findFiles(body, path, input, ["FileAttachment"]),
    imports: findImports(body, path, input),
    expression: !!expression,
    async: findAwaits(body).length > 0,
    input
  };
}
function parseProgram(input, params) {
  const body = Parser.parse(input, acornOptions);
  if (params)
    checkParams(body, input, params);
  return body;
}
function maybeParseExpression(input) {
  const parser = new Parser(acornOptions, input, 0);
  parser.nextToken();
  try {
    const node = parser.parseExpression();
    return parser.type === tokTypes.eof ? node : null;
  } catch {
    return null;
  }
}
export {
  acornOptions,
  maybeParseExpression,
  parseJavaScript,
  parseProgram
};
