/**
 *
 * A `NetworkAdapter` which uses [BroadcastChannel](https://developer.mozilla.org/en-US/docs/Web/API/BroadcastChannel)
 * to communicate with other peers in the same browser tab. This is a bit of a
 * hack because `NetworkAdapter`s are supposed to be used as point to
 * point communication channels. To get around this the {@link BroadcastChannelNetworkAdapter}
 * broadcasts messages to all peers and then filters out messages not intended
 * for the current peer. This is quite inefficient as messages get duplicated
 * for every peer in the tab, but as it's all local communication anyway
 * it's not too bad. If efficiency is becoming an issue you can switch to
 * `automerge-repo-network-messagechannel`.
 *
 * @module
 *
 */
import { NetworkAdapter, } from "@automerge/automerge-repo/slim";
export class BroadcastChannelNetworkAdapter extends NetworkAdapter {
    #broadcastChannel;
    #options;
    constructor(options) {
        super();
        this.#options = { channelName: "broadcast", ...(options ?? {}) };
        this.#broadcastChannel = new BroadcastChannel(this.#options.channelName);
    }
    connect(peerId, peerMetadata) {
        this.peerId = peerId;
        this.peerMetadata = peerMetadata;
        this.#broadcastChannel.addEventListener("message", (e) => {
            const message = e.data;
            if ("targetId" in message && message.targetId !== this.peerId) {
                return;
            }
            const { senderId, type } = message;
            switch (type) {
                case "arrive":
                    {
                        const { peerMetadata } = message;
                        this.#broadcastChannel.postMessage({
                            senderId: this.peerId,
                            targetId: senderId,
                            type: "welcome",
                            peerMetadata: this.peerMetadata,
                        });
                        this.#announceConnection(senderId, peerMetadata);
                    }
                    break;
                case "welcome":
                    {
                        const { peerMetadata } = message;
                        this.#announceConnection(senderId, peerMetadata);
                    }
                    break;
                default:
                    if (!("data" in message)) {
                        this.emit("message", message);
                    }
                    else {
                        const data = message.data;
                        this.emit("message", {
                            ...message,
                            data: new Uint8Array(data),
                        });
                    }
                    break;
            }
        });
        this.#broadcastChannel.postMessage({
            senderId: this.peerId,
            type: "arrive",
            peerMetadata,
        });
        this.emit("ready", { network: this });
    }
    #announceConnection(peerId, peerMetadata) {
        this.emit("peer-candidate", { peerId, peerMetadata });
    }
    send(message) {
        if ("data" in message) {
            this.#broadcastChannel.postMessage({
                ...message,
                data: message.data
                    ? message.data.buffer.slice(message.data.byteOffset, message.data.byteOffset + message.data.byteLength)
                    : undefined,
            });
        }
        else {
            this.#broadcastChannel.postMessage(message);
        }
    }
    disconnect() {
        // TODO:
        throw new Error("Unimplemented: leave on BroadcastChannelNetworkAdapter");
    }
}
