/**
 * # The next API
 *
 * This module contains new features we are working on which are backwards
 * incompatible with the current API of Automerge. This module will become the
 * API of the next major version of Automerge
 *
 * ## Differences from stable
 *
 * In the stable API text objects are represented using the {@link Text} class.
 * This means you must decide up front whether your string data might need
 * concurrent merges in the future and if you change your mind you have to
 * figure out how to migrate your data. In the unstable API the `Text` class is
 * gone and all `string`s are represented using the text CRDT, allowing for
 * concurrent changes. Modifying a string is done using the {@link splice}
 * function. You can still access the old behaviour of strings which do not
 * support merging behaviour via the {@link RawString} class.
 *
 * This leads to the following differences from `stable`:
 *
 * * There is no `unstable.Text` class, all strings are text objects
 * * Reading strings in an `unstable` document is the same as reading any other
 *   javascript string
 * * To modify strings in an `unstable` document use {@link splice}
 * * The {@link AutomergeValue} type does not include the {@link Text}
 *   class but the  {@link RawString} class is included in the {@link ScalarValue}
 *   type
 *
 * ## CHANGELOG
 * * Rename this module to `next` to reflect our increased confidence in it
 *   and stability commitment to it
 * * Introduce this module to expose the new API which has no `Text` class
 *
 *
 * @module
 */
export { Counter, type Doc, Int, Uint, Float64, type Patch, type PatchCallback, type Mark, type MarkSet, type MarkRange, type MarkValue, type AutomergeValue, type ScalarValue, type PatchSource, type PatchInfo, } from "./next_types.js";
import type { Cursor, CursorPosition, MoveCursor, Mark, MarkSet, MarkRange, MarkValue } from "./next_types.js";
import { type PatchCallback } from "./stable.js";
import { type UnstableConflicts as Conflicts } from "./conflicts.js";
export type { PutPatch, DelPatch, SpliceTextPatch, InsertPatch, IncPatch, MarkPatch, SyncMessage, Heads, Cursor, Span, } from "./wasm_types.js";
import type { Span, MaterializeValue } from "./wasm_types.js";
export type { ActorId, Change, ChangeOptions, Prop, DecodedChange, DecodedSyncMessage, ApplyOptions, ChangeFn, ChangeAtResult, MaterializeValue, SyncState, } from "./stable.js";
export { view, free, getHeads, change, changeAt, emptyChange, loadIncremental, saveIncremental, save, merge, getActorId, getLastLocalChange, getChanges, getAllChanges, applyChanges, getHistory, equals, encodeSyncState, decodeSyncState, generateSyncMessage, receiveSyncMessage, hasOurChanges, initSyncState, encodeChange, decodeChange, encodeSyncMessage, decodeSyncMessage, getMissingDeps, dump, toJS, isAutomerge, getObjectId, diff, insertAt, deleteAt, saveSince, initializeWasm, initializeBase64Wasm, wasmInitialized, isWasmInitialized, hasHeads, topoHistoryTraversal, inspectChange, stats, } from "./stable.js";
export { isRawString } from "./proxies.js";
export type InitOptions<T> = {
    /** The actor ID to use for this document, a random one will be generated if `null` is passed */
    actor?: ActorId;
    freeze?: boolean;
    /** A callback which will be called with the initial patch once the document has finished loading */
    patchCallback?: PatchCallback<T>;
    /** @hidden */
    unchecked?: boolean;
    /** Allow loading a document with missing changes */
    allowMissingChanges?: boolean;
    /** Whether to convert raw string to text objects
     *
     * @remarks
     * This is useful if you have some documents which were created using the older API which represented
     * text as the `Text` class and you are migrating to the new API where text is just a `string`. In
     * this case the strings from the old document will appear as `RawString`s in the new document. This
     * option will convert those `RawString`s to `Text` objects. This conversion is achieved by rewriting
     * all the old string fields to new text fields
     **/
    convertRawStringsToText?: boolean;
};
import { ActorId, Doc } from "./stable.js";
import * as stable from "./stable.js";
export { RawString } from "./raw_string.js";
/** @hidden */
export declare const getBackend: typeof stable.getBackend;
/**
 * Create a new automerge document
 *
 * @typeParam T - The type of value contained in the document. This will be the
 *     type that is passed to the change closure in {@link change}
 * @param _opts - Either an actorId or an {@link InitOptions} (which may
 *     contain an actorId). If this is null the document will be initialised with a
 *     random actor ID
 */
export declare function init<T>(_opts?: ActorId | InitOptions<T>): Doc<T>;
/**
 * Make a full writable copy of an automerge document
 *
 * @remarks
 * Unlike {@link view} this function makes a full copy of the memory backing
 * the document and can thus be passed to {@link change}. It also generates a
 * new actor ID so that changes made in the new document do not create duplicate
 * sequence numbers with respect to the old document. If you need control over
 * the actor ID which is generated you can pass the actor ID as the second
 * argument
 *
 * @typeParam T - The type of the value contained in the document
 * @param doc - The document to clone
 * @param _opts - Either an actor ID to use for the new doc or an {@link InitOptions}
 */
export declare function clone<T>(doc: Doc<T>, _opts?: ActorId | InitOptions<T>): Doc<T>;
/**
 * Create an automerge document from a POJO
 *
 * @param initialState - The initial state which will be copied into the document
 * @typeParam T - The type of the value passed to `from` _and_ the type the resulting document will contain
 * @typeParam actor - The actor ID of the resulting document, if this is null a random actor ID will be used
 *
 * @example
 * ```
 * const doc = automerge.from({
 *     tasks: [
 *         {description: "feed dogs", done: false}
 *     ]
 * })
 * ```
 */
export declare function from<T extends Record<string, unknown>>(initialState: T | Doc<T>, _opts?: ActorId | InitOptions<T>): Doc<T>;
/**
 * Load an automerge document from a compressed document produce by {@link save}
 *
 * @typeParam T - The type of the value which is contained in the document.
 *                Note that no validation is done to make sure this type is in
 *                fact the type of the contained value so be a bit careful
 * @param data  - The compressed document
 * @param _opts - Either an actor ID or some {@link InitOptions}, if the actor
 *                ID is null a random actor ID will be created
 *
 * Note that `load` will throw an error if passed incomplete content (for
 * example if you are receiving content over the network and don't know if you
 * have the complete document yet). If you need to handle incomplete content use
 * {@link init} followed by {@link loadIncremental}.
 */
export declare function load<T>(data: Uint8Array, _opts?: ActorId | InitOptions<T>): Doc<T>;
/**
 * Modify a string
 *
 * @typeParam T - The type of the value contained in the document
 * @param doc - The document to modify
 * @param path - The path to the string to modify
 * @param index - The position (as a {@link Cursor} or index) to edit.
 *   If a cursor is used then the edit happens such that the cursor will
 *   now point to the end of the newText, so you can continue to reuse
 *   the same cursor for multiple calls to splice.
 * @param del - The number of code units to delete, a positive number
 *   deletes N characters after the cursor, a negative number deletes
 *   N characters before the cursor.
 * @param newText - The string to insert (if any).
 */
export declare function splice<T>(doc: Doc<T>, path: stable.Prop[], index: number | Cursor, del: number, newText?: string): void;
/**
 * Update the value of a string
 *
 * @typeParam T - The type of the value contained in the document
 * @param doc - The document to modify
 * @param path - The path to the string to modify
 * @param newText - The new text to update the value to
 *
 * @remarks
 * This will calculate a diff between the current value and the new value and
 * then convert that diff into calls to {@link splice}. This will produce results
 * which don't merge as well as directly capturing the user input actions, but
 * sometimes it's not possible to capture user input and this is the best you
 * can do.
 *
 * This is an experimental API and may change in the future.
 *
 * @beta
 */
export declare function updateText(doc: Doc<unknown>, path: stable.Prop[], newText: string): void;
/**
 * Return the text + block markers at a given path
 *
 * @remarks
 * Rich text in automerge is represented as a sequence of characters with block
 * markers appearing inline with the text, and inline formatting spans overlaid
 * on the whole sequence. Block markers are normal automerge maps, but they are
 * only visible via either the {@link block} function or the {@link spans}
 * function. This function returns the current state of the spans
 */
export declare function spans<T>(doc: Doc<T>, path: stable.Prop[]): Span[];
/**
 * Get the block marker at the given index
 */
export declare function block<T>(doc: Doc<T>, path: stable.Prop[], index: number | Cursor): {
    [key: string]: MaterializeValue;
} | null;
/**
 * Insert a new block marker at the given index
 */
export declare function splitBlock<T>(doc: Doc<T>, path: stable.Prop[], index: number | Cursor, block: {
    [key: string]: MaterializeValue;
}): void;
/**
 * Delete the block marker at the given index
 */
export declare function joinBlock<T>(doc: Doc<T>, path: stable.Prop[], index: number | Cursor): void;
/**
 * Update the block marker at the given index
 */
export declare function updateBlock<T>(doc: Doc<T>, path: stable.Prop[], index: number | Cursor, block: {
    [key: string]: MaterializeValue;
}): void;
/**
 * Update the spans at the given path
 *
 * @remarks
 * Like {@link updateText} this will diff `newSpans` against the current state
 * of the text at `path` and perform a reasonably minimal number of operations
 * required to update the spans to the new state.
 */
export declare function updateSpans<T>(doc: Doc<T>, path: stable.Prop[], newSpans: Span[]): void;
/**
 * Returns a cursor for the given position in a string.
 *
 * @remarks
 * A cursor represents a relative position, "before character X",
 * rather than an absolute position. As the document is edited, the
 * cursor remains stable relative to its context, just as you'd expect
 * from a cursor in a concurrent text editor.
 *
 * The string representation is shareable, and so you can use this both
 * to edit the document yourself (using {@link splice}) or to share multiple
 * collaborator's current cursor positions over the network.
 *
 * The cursor's `position` can be an index in the string, `'start'` or `'end'`.
 * - `'start'` ensures this cursor always resolves to `0`
 * - `'end'` ensures this cursor always resolves to `string.length`
 *
 * Start cursors can be created by passing any negative number in `position`.
 *
 * End cursors can be created by passing a number `>= string.length` in `position`.
 *
 * `move` determines the position the cursor resolves to if the character at
 * `index` is removed:
 * - `'after'` causes the cursor to resolve towards `string.length`
 * - `'before'` causes the cursor to resolve towards `0`
 *
 * `move` is `'after'` by default.
 *
 * @typeParam T - The type of the value contained in the document
 * @param doc - The document
 * @param path - The path to the string
 * @param position - The position of the cursor, either an index, `'start'` or `'end'`
 * @param move - The direction the cursor should resolve to, defaults to 'after'
 */
export declare function getCursor<T>(doc: Doc<T>, path: stable.Prop[], position: CursorPosition, move?: MoveCursor): Cursor;
/**
 * Returns the current index of the cursor.
 *
 * @typeParam T - The type of the value contained in the document
 *
 * @param doc - The document
 * @param path - The path to the string
 * @param index - The cursor
 */
export declare function getCursorPosition<T>(doc: Doc<T>, path: stable.Prop[], cursor: Cursor): number;
export declare function mark<T>(doc: Doc<T>, path: stable.Prop[], range: MarkRange, name: string, value: MarkValue): void;
export declare function unmark<T>(doc: Doc<T>, path: stable.Prop[], range: MarkRange, name: string): void;
export declare function marks<T>(doc: Doc<T>, path: stable.Prop[]): Mark[];
export declare function marksAt<T>(doc: Doc<T>, path: stable.Prop[], index: number): MarkSet;
/**
 * Get the conflicts associated with a property
 *
 * The values of properties in a map in automerge can be conflicted if there
 * are concurrent "put" operations to the same key. Automerge chooses one value
 * arbitrarily (but deterministically, any two nodes who have the same set of
 * changes will choose the same value) from the set of conflicting values to
 * present as the value of the key.
 *
 * Sometimes you may want to examine these conflicts, in this case you can use
 * {@link getConflicts} to get the conflicts for the key.
 *
 * @example
 * ```
 * import * as automerge from "@automerge/automerge"
 *
 * type Profile = {
 *     pets: Array<{name: string, type: string}>
 * }
 *
 * let doc1 = automerge.init<Profile>("aaaa")
 * doc1 = automerge.change(doc1, d => {
 *     d.pets = [{name: "Lassie", type: "dog"}]
 * })
 * let doc2 = automerge.init<Profile>("bbbb")
 * doc2 = automerge.merge(doc2, automerge.clone(doc1))
 *
 * doc2 = automerge.change(doc2, d => {
 *     d.pets[0].name = "Beethoven"
 * })
 *
 * doc1 = automerge.change(doc1, d => {
 *     d.pets[0].name = "Babe"
 * })
 *
 * const doc3 = automerge.merge(doc1, doc2)
 *
 * // Note that here we pass `doc3.pets`, not `doc3`
 * let conflicts = automerge.getConflicts(doc3.pets[0], "name")
 *
 * // The two conflicting values are the keys of the conflicts object
 * assert.deepEqual(Object.values(conflicts), ["Babe", "Beethoven"])
 * ```
 */
export declare function getConflicts<T>(doc: Doc<T>, prop: stable.Prop): Conflicts | undefined;
