import { Box2d } from "../../../../primitives/Box2d.mjs";
import { Matrix2d } from "../../../../primitives/Matrix2d.mjs";
import { Vec2d } from "../../../../primitives/Vec2d.mjs";
import { intersectCirclePolygon, intersectCirclePolyline } from "../../../../primitives/intersect.mjs";
import {
  PI,
  PI2,
  clockwiseAngleDist,
  counterClockwiseAngleDist,
  isSafeFloat
} from "../../../../primitives/utils.mjs";
import {
  BOUND_ARROW_OFFSET,
  MIN_ARROW_LENGTH,
  STROKE_SIZES,
  WAY_TOO_BIG_ARROW_BEND_FACTOR,
  getArrowTerminalsInArrowSpace,
  getBoundShapeInfoForTerminal
} from "./shared.mjs";
import { getStraightArrowInfo } from "./straight-arrow.mjs";
function getCurvedArrowInfo(editor, shape, extraBend = 0) {
  const { arrowheadEnd, arrowheadStart } = shape.props;
  const bend = shape.props.bend + extraBend;
  if (Math.abs(bend) > Math.abs(shape.props.bend * WAY_TOO_BIG_ARROW_BEND_FACTOR)) {
    return getStraightArrowInfo(editor, shape);
  }
  const terminalsInArrowSpace = getArrowTerminalsInArrowSpace(editor, shape);
  const med = Vec2d.Med(terminalsInArrowSpace.start, terminalsInArrowSpace.end);
  const u = Vec2d.Sub(terminalsInArrowSpace.end, terminalsInArrowSpace.start).uni();
  const middle = Vec2d.Add(med, u.per().mul(-bend));
  const startShapeInfo = getBoundShapeInfoForTerminal(editor, shape.props.start);
  const endShapeInfo = getBoundShapeInfoForTerminal(editor, shape.props.end);
  const a = terminalsInArrowSpace.start.clone();
  const b = terminalsInArrowSpace.end.clone();
  const c = middle.clone();
  const isClockwise = shape.props.bend < 0;
  const distFn = isClockwise ? clockwiseAngleDist : counterClockwiseAngleDist;
  const handleArc = getArcInfo(a, b, c);
  const handle_aCA = Vec2d.Angle(handleArc.center, a);
  const handle_aCB = Vec2d.Angle(handleArc.center, b);
  const handle_dAB = distFn(handle_aCA, handle_aCB);
  if (handleArc.length === 0 || handleArc.size === 0 || !isSafeFloat(handleArc.length) || !isSafeFloat(handleArc.size)) {
    return getStraightArrowInfo(editor, shape);
  }
  const tempA = a.clone();
  const tempB = b.clone();
  const tempC = c.clone();
  const arrowPageTransform = editor.getShapePageTransform(shape);
  let offsetA = 0;
  let offsetB = 0;
  if (startShapeInfo && !startShapeInfo.isExact) {
    const startInPageSpace = Matrix2d.applyToPoint(arrowPageTransform, tempA);
    const centerInPageSpace = Matrix2d.applyToPoint(arrowPageTransform, handleArc.center);
    const endInPageSpace = Matrix2d.applyToPoint(arrowPageTransform, tempB);
    const inverseTransform = Matrix2d.Inverse(startShapeInfo.transform);
    const startInStartShapeLocalSpace = Matrix2d.applyToPoint(inverseTransform, startInPageSpace);
    const centerInStartShapeLocalSpace = Matrix2d.applyToPoint(inverseTransform, centerInPageSpace);
    const endInStartShapeLocalSpace = Matrix2d.applyToPoint(inverseTransform, endInPageSpace);
    const { isClosed } = startShapeInfo;
    const fn = isClosed ? intersectCirclePolygon : intersectCirclePolyline;
    let point;
    let intersections = fn(centerInStartShapeLocalSpace, handleArc.radius, startShapeInfo.outline);
    if (intersections) {
      const angleToStart = centerInStartShapeLocalSpace.angle(startInStartShapeLocalSpace);
      const angleToEnd = centerInStartShapeLocalSpace.angle(endInStartShapeLocalSpace);
      const dAB2 = distFn(angleToStart, angleToEnd);
      intersections = intersections.filter(
        (pt) => distFn(angleToStart, centerInStartShapeLocalSpace.angle(pt)) <= dAB2
      );
      const targetDist = dAB2 * 0.25;
      intersections.sort(
        isClosed ? (p0, p1) => Math.abs(distFn(angleToStart, centerInStartShapeLocalSpace.angle(p0)) - targetDist) < Math.abs(distFn(angleToStart, centerInStartShapeLocalSpace.angle(p1)) - targetDist) ? -1 : 1 : (p0, p1) => distFn(angleToStart, centerInStartShapeLocalSpace.angle(p0)) < distFn(angleToStart, centerInStartShapeLocalSpace.angle(p1)) ? -1 : 1
      );
      point = intersections[0] ?? (isClosed ? void 0 : startInStartShapeLocalSpace);
    } else {
      point = isClosed ? void 0 : startInStartShapeLocalSpace;
    }
    if (point) {
      tempA.setTo(
        editor.getPointInShapeSpace(shape, Matrix2d.applyToPoint(startShapeInfo.transform, point))
      );
      startShapeInfo.didIntersect = true;
      if (arrowheadStart !== "none") {
        offsetA = BOUND_ARROW_OFFSET + STROKE_SIZES[shape.props.size] / 2 + ("size" in startShapeInfo.shape.props ? STROKE_SIZES[startShapeInfo.shape.props.size] / 2 : 0);
      }
    }
  }
  if (endShapeInfo && !endShapeInfo.isExact) {
    const startInPageSpace = Matrix2d.applyToPoint(arrowPageTransform, tempA);
    const endInPageSpace = Matrix2d.applyToPoint(arrowPageTransform, tempB);
    const centerInPageSpace = Matrix2d.applyToPoint(arrowPageTransform, handleArc.center);
    const inverseTransform = Matrix2d.Inverse(endShapeInfo.transform);
    const startInEndShapeLocalSpace = Matrix2d.applyToPoint(inverseTransform, startInPageSpace);
    const centerInEndShapeLocalSpace = Matrix2d.applyToPoint(inverseTransform, centerInPageSpace);
    const endInEndShapeLocalSpace = Matrix2d.applyToPoint(inverseTransform, endInPageSpace);
    const isClosed = endShapeInfo.isClosed;
    const fn = isClosed ? intersectCirclePolygon : intersectCirclePolyline;
    let point;
    let intersections = fn(centerInEndShapeLocalSpace, handleArc.radius, endShapeInfo.outline);
    if (intersections) {
      const angleToStart = centerInEndShapeLocalSpace.angle(startInEndShapeLocalSpace);
      const angleToEnd = centerInEndShapeLocalSpace.angle(endInEndShapeLocalSpace);
      const dAB2 = distFn(angleToStart, angleToEnd);
      const targetDist = dAB2 * 0.75;
      intersections = intersections.filter(
        (pt) => distFn(angleToStart, centerInEndShapeLocalSpace.angle(pt)) <= dAB2
      );
      intersections.sort(
        isClosed ? (p0, p1) => Math.abs(distFn(angleToStart, centerInEndShapeLocalSpace.angle(p0)) - targetDist) < Math.abs(distFn(angleToStart, centerInEndShapeLocalSpace.angle(p1)) - targetDist) ? -1 : 1 : (p0, p1) => distFn(angleToStart, centerInEndShapeLocalSpace.angle(p0)) < distFn(angleToStart, centerInEndShapeLocalSpace.angle(p1)) ? -1 : 1
      );
      if (intersections[0]) {
        point = intersections[0];
      } else {
        point = isClosed ? void 0 : endInEndShapeLocalSpace;
      }
    } else {
      point = isClosed ? void 0 : endInEndShapeLocalSpace;
    }
    if (point) {
      tempB.setTo(
        editor.getPointInShapeSpace(shape, Matrix2d.applyToPoint(endShapeInfo.transform, point))
      );
      endShapeInfo.didIntersect = true;
      if (arrowheadEnd !== "none") {
        offsetB = BOUND_ARROW_OFFSET + STROKE_SIZES[shape.props.size] / 2 + ("size" in endShapeInfo.shape.props ? STROKE_SIZES[endShapeInfo.shape.props.size] / 2 : 0);
      }
    }
  }
  let aCA = Vec2d.Angle(handleArc.center, tempA);
  let aCB = Vec2d.Angle(handleArc.center, tempB);
  let dAB = distFn(aCA, aCB);
  let lAB = dAB * handleArc.radius;
  const tA = tempA.clone();
  const tB = tempB.clone();
  if (offsetA !== 0) {
    const n = offsetA / lAB * (isClockwise ? 1 : -1);
    const u2 = Vec2d.FromAngle(aCA + dAB * n);
    tA.setTo(handleArc.center).add(u2.mul(handleArc.radius));
  }
  if (offsetB !== 0) {
    const n = offsetB / lAB * (isClockwise ? -1 : 1);
    const u2 = Vec2d.FromAngle(aCB + dAB * n);
    tB.setTo(handleArc.center).add(u2.mul(handleArc.radius));
  }
  const distAB = Vec2d.Dist(tA, tB);
  if (distAB < MIN_ARROW_LENGTH) {
    if (offsetA !== 0 && offsetB !== 0) {
      offsetA *= -1.5;
      offsetB *= -1.5;
    } else if (offsetA !== 0) {
      offsetA *= -2;
    } else if (offsetB !== 0) {
      offsetB *= -2;
    } else {
      if (distAB < 10) {
        if (startShapeInfo)
          offsetA = -(10 - distAB);
        else if (endShapeInfo)
          offsetB = -(10 - distAB);
      }
    }
  }
  if (offsetA !== 0) {
    const n = offsetA / lAB * (isClockwise ? 1 : -1);
    const u2 = Vec2d.FromAngle(aCA + dAB * n);
    tempA.setTo(handleArc.center).add(u2.mul(handleArc.radius));
  }
  if (offsetB !== 0) {
    const n = offsetB / lAB * (isClockwise ? -1 : 1);
    const u2 = Vec2d.FromAngle(aCB + dAB * n);
    tempB.setTo(handleArc.center).add(u2.mul(handleArc.radius));
  }
  if (startShapeInfo && endShapeInfo && !startShapeInfo.isExact && !endShapeInfo.isExact) {
    aCA = Vec2d.Angle(handleArc.center, tempA);
    aCB = Vec2d.Angle(handleArc.center, tempB);
    dAB = distFn(aCA, aCB);
    lAB = dAB * handleArc.radius;
    if (startShapeInfo.shape === endShapeInfo.shape) {
      if (lAB < 100) {
        tempA.setTo(a);
        tempB.setTo(b);
        tempC.setTo(c);
      }
    } else {
      if (startShapeInfo && !startShapeInfo.didIntersect) {
        tempA.setTo(a);
      }
      if (endShapeInfo && !endShapeInfo.didIntersect || distFn(handle_aCA, aCA) > distFn(handle_aCA, aCB)) {
        const n = Math.min(0.9, MIN_ARROW_LENGTH / lAB) * (isClockwise ? 1 : -1);
        const u2 = Vec2d.FromAngle(aCA + dAB * n);
        tempB.setTo(handleArc.center).add(u2.mul(handleArc.radius));
      }
    }
  }
  placeCenterHandle(
    handleArc.center,
    handleArc.radius,
    tempA,
    tempB,
    tempC,
    handle_dAB,
    isClockwise
  );
  if (tempA.equals(tempB)) {
    tempA.setTo(tempC.clone().addXY(1, 1));
    tempB.setTo(tempC.clone().subXY(1, 1));
  }
  a.setTo(tempA);
  b.setTo(tempB);
  c.setTo(tempC);
  const bodyArc = getArcInfo(a, b, c);
  return {
    isStraight: false,
    start: {
      point: a,
      handle: terminalsInArrowSpace.start,
      arrowhead: shape.props.arrowheadStart
    },
    end: {
      point: b,
      handle: terminalsInArrowSpace.end,
      arrowhead: shape.props.arrowheadEnd
    },
    middle: c,
    handleArc,
    bodyArc,
    isValid: bodyArc.length !== 0 && isFinite(bodyArc.center.x) && isFinite(bodyArc.center.y)
  };
}
function getCurvedArrowHandlePath(info) {
  const {
    start,
    end,
    handleArc: { radius, largeArcFlag, sweepFlag }
  } = info;
  return `M${start.handle.x},${start.handle.y} A${radius} ${radius} 0 ${largeArcFlag} ${sweepFlag} ${end.handle.x},${end.handle.y}`;
}
function getSolidCurvedArrowPath(info) {
  const {
    start,
    end,
    bodyArc: { radius, largeArcFlag, sweepFlag }
  } = info;
  return `M${start.point.x},${start.point.y} A${radius} ${radius} 0 ${largeArcFlag} ${sweepFlag} ${end.point.x},${end.point.y}`;
}
function getPointOnArc(center, radius, startAngle, size, t) {
  const angle = startAngle + size * t;
  return new Vec2d(center.x + radius * Math.cos(angle), center.y + radius * Math.sin(angle));
}
function getArcBoundingBox(center, radius, start, size) {
  let minX = Infinity;
  let minY = Infinity;
  let maxX = -Infinity;
  let maxY = -Infinity;
  const startAngle = Vec2d.Angle(center, start);
  for (let i = 0; i < 20; i++) {
    const angle = startAngle + size * (i / 19);
    const x = center.x + radius * Math.cos(angle);
    const y = center.y + radius * Math.sin(angle);
    minX = Math.min(x, minX);
    minY = Math.min(y, minY);
    maxX = Math.max(x, maxX);
    maxY = Math.max(y, maxY);
  }
  return new Box2d(minX, minY, maxX - minX, maxY - minY);
}
function getArcInfo(a, b, c) {
  const u = -2 * (a.x * (b.y - c.y) - a.y * (b.x - c.x) + b.x * c.y - c.x * b.y);
  const center = {
    x: ((a.x * a.x + a.y * a.y) * (c.y - b.y) + (b.x * b.x + b.y * b.y) * (a.y - c.y) + (c.x * c.x + c.y * c.y) * (b.y - a.y)) / u,
    y: ((a.x * a.x + a.y * a.y) * (b.x - c.x) + (b.x * b.x + b.y * b.y) * (c.x - a.x) + (c.x * c.x + c.y * c.y) * (a.x - b.x)) / u
  };
  const radius = Vec2d.Dist(center, a);
  const sweepFlag = +Vec2d.Clockwise(a, c, b);
  const ab = Math.hypot(a.y - b.y, a.x - b.x);
  const bc = Math.hypot(b.y - c.y, b.x - c.x);
  const ca = Math.hypot(c.y - a.y, c.x - a.x);
  const theta = Math.acos((bc * bc + ca * ca - ab * ab) / (2 * bc * ca)) * 2;
  const largeArcFlag = +(PI > theta);
  const size = (PI2 - theta) * (sweepFlag ? 1 : -1);
  const length = size * radius;
  return {
    center,
    radius,
    size,
    length,
    largeArcFlag,
    sweepFlag
  };
}
function placeCenterHandle(center, radius, tempA, tempB, tempC, originalArcLength, isClockwise) {
  const aCA = Vec2d.Angle(center, tempA);
  const aCB = Vec2d.Angle(center, tempB);
  let dAB = clockwiseAngleDist(aCA, aCB);
  if (!isClockwise)
    dAB = PI2 - dAB;
  const n = 0.5 * (isClockwise ? 1 : -1);
  const u = Vec2d.FromAngle(aCA + dAB * n);
  tempC.setTo(center).add(u.mul(radius));
  if (dAB > originalArcLength) {
    tempC.rotWith(center, PI);
    const t = tempB.clone();
    tempB.setTo(tempA);
    tempA.setTo(t);
  }
}
export {
  getArcBoundingBox,
  getArcInfo,
  getCurvedArrowHandlePath,
  getCurvedArrowInfo,
  getPointOnArc,
  getSolidCurvedArrowPath
};
//# sourceMappingURL=curved-arrow.mjs.map
