import { isShapeId } from "@tldraw/tlschema";
import { structuredClone } from "@tldraw/utils";
import { Matrix2d } from "../primitives/Matrix2d.mjs";
import { canonicalizeRotation } from "../primitives/utils.mjs";
import { Vec2d } from "../primitives/Vec2d.mjs";
function getRotationSnapshot({ editor }) {
  const {
    selectionRotation,
    selectionRotatedPageBounds: selectionBounds,
    inputs: { originPagePoint },
    selectedShapes
  } = editor;
  if (!selectionBounds) {
    return null;
  }
  const selectionPageCenter = selectionBounds.center.clone().rotWith(selectionBounds.point, selectionRotation);
  return {
    selectionPageCenter,
    initialCursorAngle: selectionPageCenter.angle(originPagePoint),
    initialSelectionRotation: selectionRotation,
    shapeSnapshots: selectedShapes.map((shape) => ({
      shape: structuredClone(shape),
      initialPagePoint: editor.getShapePageTransform(shape.id).point()
    }))
  };
}
function applyRotationToSnapshotShapes({
  delta,
  editor,
  snapshot,
  stage
}) {
  const { selectionPageCenter, shapeSnapshots } = snapshot;
  editor.updateShapes(
    shapeSnapshots.map(({ shape, initialPagePoint }) => {
      const parentTransform = isShapeId(shape.parentId) ? editor.getShapePageTransform(shape.parentId) : Matrix2d.Identity();
      const newPagePoint = Vec2d.RotWith(initialPagePoint, selectionPageCenter, delta);
      const newLocalPoint = Matrix2d.applyToPoint(
        // use the current parent transform in case it has moved/resized since the start
        // (e.g. if rotating a shape at the edge of a group)
        Matrix2d.Inverse(parentTransform),
        newPagePoint
      );
      const newRotation = canonicalizeRotation(shape.rotation + delta);
      return {
        id: shape.id,
        type: shape.type,
        x: newLocalPoint.x,
        y: newLocalPoint.y,
        rotation: newRotation
      };
    })
  );
  const changes = [];
  shapeSnapshots.forEach(({ shape }) => {
    const current = editor.getShape(shape.id);
    if (!current)
      return;
    const util = editor.getShapeUtil(shape);
    if (stage === "start" || stage === "one-off") {
      const changeStart = util.onRotateStart?.(shape);
      if (changeStart)
        changes.push(changeStart);
    }
    const changeUpdate = util.onRotate?.(shape, current);
    if (changeUpdate)
      changes.push(changeUpdate);
    if (stage === "end" || stage === "one-off") {
      const changeEnd = util.onRotateEnd?.(shape, current);
      if (changeEnd)
        changes.push(changeEnd);
    }
  });
  if (changes.length > 0) {
    editor.updateShapes(changes);
  }
}
export {
  applyRotationToSnapshotShapes,
  getRotationSnapshot
};
//# sourceMappingURL=rotation.mjs.map
