import { createShapeId } from "@tldraw/tlschema";
import { Vec2d } from "../../../../primitives/Vec2d.mjs";
import { StateNode } from "../../StateNode.mjs";
class Pointing extends StateNode {
  static id = "pointing";
  markId = "";
  wasFocusedOnEnter = false;
  onEnter = () => {
    const { isMenuOpen } = this.editor;
    this.wasFocusedOnEnter = !isMenuOpen;
  };
  onPointerMove = (info) => {
    if (this.editor.inputs.isDragging) {
      const { originPagePoint } = this.editor.inputs;
      const shapeType = this.parent.shapeType;
      const id = createShapeId();
      this.markId = `creating:${id}`;
      this.editor.mark(this.markId);
      this.editor.createShapes([
        {
          id,
          type: shapeType,
          x: originPagePoint.x,
          y: originPagePoint.y,
          props: {
            w: 1,
            h: 1
          }
        }
      ]).select(id);
      this.editor.setCurrentTool("select.resizing", {
        ...info,
        target: "selection",
        handle: "bottom_right",
        isCreating: true,
        creationCursorOffset: { x: 1, y: 1 },
        onInteractionEnd: this.parent.id
      });
    }
  };
  onPointerUp = () => {
    this.complete();
  };
  onCancel = () => {
    this.cancel();
  };
  onComplete = () => {
    this.complete();
  };
  onInterrupt = () => {
    this.cancel();
  };
  complete() {
    const { originPagePoint } = this.editor.inputs;
    if (!this.wasFocusedOnEnter) {
      return;
    }
    this.editor.mark(this.markId);
    const shapeType = this.parent.shapeType;
    const id = createShapeId();
    this.editor.mark(this.markId);
    this.editor.createShapes([
      {
        id,
        type: shapeType,
        x: originPagePoint.x,
        y: originPagePoint.y
      }
    ]);
    const shape = this.editor.getShape(id);
    const { w, h } = this.editor.getShapeUtil(shape).getDefaultProps();
    const delta = new Vec2d(w / 2, h / 2);
    const parentTransform = this.editor.getShapeParentTransform(shape);
    if (parentTransform)
      delta.rot(-parentTransform.rotation());
    this.editor.updateShapes([
      {
        id,
        type: shapeType,
        x: shape.x - delta.x,
        y: shape.y - delta.y
      }
    ]);
    this.editor.setSelectedShapes([id]);
    if (this.editor.instanceState.isToolLocked) {
      this.parent.transition("idle", {});
    } else {
      this.editor.setCurrentTool("select.idle");
    }
  }
  cancel() {
    this.parent.transition("idle", {});
  }
}
export {
  Pointing
};
//# sourceMappingURL=Pointing.mjs.map
