/**
 * A `NetworkAdapter` which uses [`MessageChannel`](https://developer.mozilla.org/en-US/docs/Web/API/MessageChannel)
 * to communicate with other peers. This is useful for communicating between
 * browser tabs and web workers (including shared workers).
 *
 * @module
 */
import { NetworkAdapter, } from "@automerge/automerge-repo/slim";
import { StrongMessagePortRef } from "./StrongMessagePortRef.js";
import { WeakMessagePortRef } from "./WeakMessagePortRef.js";
import debug from "debug";
const log = debug("automerge-repo:messagechannel");
export class MessageChannelNetworkAdapter extends NetworkAdapter {
    channels = {};
    /** @hidden */
    messagePortRef;
    #startupComplete = false;
    constructor(messagePort, config = {}) {
        super();
        const useWeakRef = config.useWeakRef ?? false;
        this.messagePortRef = useWeakRef
            ? new WeakMessagePortRef(messagePort)
            : new StrongMessagePortRef(messagePort);
    }
    connect(peerId, peerMetadata) {
        log("messageport connecting");
        this.peerId = peerId;
        this.peerMetadata = peerMetadata;
        this.messagePortRef.start();
        this.messagePortRef.addListener("message", (e) => {
            log("message port received %o", e.data);
            const message = e.data;
            if ("targetId" in message && message.targetId !== this.peerId) {
                throw new Error("MessagePortNetwork should never receive messages for a different peer.");
            }
            const { senderId, type } = message;
            switch (type) {
                case "arrive":
                    {
                        const { peerMetadata } = message;
                        this.messagePortRef.postMessage({
                            type: "welcome",
                            senderId: this.peerId,
                            peerMetadata: this.peerMetadata,
                            targetId: senderId,
                        });
                        this.announceConnection(senderId, peerMetadata);
                    }
                    break;
                case "welcome":
                    {
                        const { peerMetadata } = message;
                        this.announceConnection(senderId, peerMetadata);
                    }
                    break;
                default:
                    if (!("data" in message)) {
                        this.emit("message", message);
                    }
                    else {
                        this.emit("message", {
                            ...message,
                            data: message.data ? new Uint8Array(message.data) : undefined,
                        });
                    }
                    break;
            }
        });
        this.messagePortRef.addListener("close", () => {
            this.emit("close");
        });
        this.messagePortRef.postMessage({
            senderId: this.peerId,
            type: "arrive",
            peerMetadata,
        });
        // Mark this messagechannel as ready after 50 ms, at this point there
        // must be something weird going on on the other end to cause us to receive
        // no response
        setTimeout(() => {
            if (!this.#startupComplete) {
                this.#startupComplete = true;
                this.emit("ready", { network: this });
            }
        }, 100);
    }
    send(message) {
        if ("data" in message) {
            const data = message.data.buffer.slice(message.data.byteOffset, message.data.byteOffset + message.data.byteLength);
            this.messagePortRef.postMessage({
                ...message,
                data,
            }, [data]);
        }
        else {
            this.messagePortRef.postMessage(message);
        }
    }
    announceConnection(peerId, peerMetadata) {
        if (!this.#startupComplete) {
            this.#startupComplete = true;
            this.emit("ready", { network: this });
        }
        this.emit("peer-candidate", { peerId, peerMetadata });
    }
    disconnect() {
        // TODO
        throw new Error("Unimplemented: leave on MessagePortNetworkAdapter");
    }
}
