import {
  EPSILON,
  StateNode,
  angleDelta,
  applyRotationToSnapshotShapes,
  degreesToRadians,
  getRotationSnapshot,
  snapAngle
} from "@tldraw/editor";
import { CursorTypeMap } from "./PointingResizeHandle.mjs";
class Rotating extends StateNode {
  static id = "rotating";
  snapshot = {};
  info = {};
  markId = "";
  onEnter = (info) => {
    this.info = info;
    this.parent.currentToolIdMask = info.onInteractionEnd;
    this.markId = "rotate start";
    this.editor.mark(this.markId);
    const snapshot = getRotationSnapshot({ editor: this.editor });
    if (!snapshot)
      return this.parent.transition("idle", this.info);
    this.snapshot = snapshot;
    this.handleStart();
  };
  onExit = () => {
    this.editor.setCursor({ type: "default", rotation: 0 });
    this.parent.currentToolIdMask = void 0;
    this.snapshot = {};
  };
  onPointerMove = () => {
    this.update();
  };
  onKeyDown = () => {
    this.update();
  };
  onKeyUp = () => {
    this.update();
  };
  onPointerUp = () => {
    this.complete();
  };
  onComplete = () => {
    this.complete();
  };
  onCancel = () => {
    this.cancel();
  };
  // ---
  update = () => {
    const newSelectionRotation = this._getRotationFromPointerPosition({
      snapToNearestDegree: false
    });
    applyRotationToSnapshotShapes({
      editor: this.editor,
      delta: newSelectionRotation,
      snapshot: this.snapshot,
      stage: "update"
    });
    this.editor.updateInstanceState({
      cursor: {
        type: CursorTypeMap[this.info.handle],
        rotation: newSelectionRotation + this.snapshot.initialSelectionRotation
      }
    });
  };
  cancel = () => {
    this.editor.bailToMark(this.markId);
    if (this.info.onInteractionEnd) {
      this.editor.setCurrentTool(this.info.onInteractionEnd, this.info);
    } else {
      this.parent.transition("idle", this.info);
    }
  };
  complete = () => {
    applyRotationToSnapshotShapes({
      editor: this.editor,
      delta: this._getRotationFromPointerPosition({ snapToNearestDegree: true }),
      snapshot: this.snapshot,
      stage: "end"
    });
    if (this.info.onInteractionEnd) {
      this.editor.setCurrentTool(this.info.onInteractionEnd, this.info);
    } else {
      this.parent.transition("idle", this.info);
    }
  };
  handleStart() {
    const newSelectionRotation = this._getRotationFromPointerPosition({
      snapToNearestDegree: false
    });
    applyRotationToSnapshotShapes({
      editor: this.editor,
      delta: this._getRotationFromPointerPosition({ snapToNearestDegree: false }),
      snapshot: this.snapshot,
      stage: "start"
    });
    this.editor.updateInstanceState({
      cursor: {
        type: CursorTypeMap[this.info.handle],
        rotation: newSelectionRotation + this.snapshot.initialSelectionRotation
      }
    });
  }
  _getRotationFromPointerPosition({ snapToNearestDegree }) {
    const {
      selectionRotatedPageBounds: selectionBounds,
      selectionRotation,
      inputs: { shiftKey, currentPagePoint }
    } = this.editor;
    const { initialCursorAngle, initialSelectionRotation } = this.snapshot;
    if (!selectionBounds)
      return initialSelectionRotation;
    const selectionPageCenter = selectionBounds.center.clone().rotWith(selectionBounds.point, selectionRotation);
    const preSnapRotationDelta = selectionPageCenter.angle(currentPagePoint) - initialCursorAngle;
    let newSelectionRotation = initialSelectionRotation + preSnapRotationDelta;
    if (shiftKey) {
      newSelectionRotation = snapAngle(newSelectionRotation, 24);
    } else if (snapToNearestDegree) {
      newSelectionRotation = Math.round(newSelectionRotation / EPSILON) * EPSILON;
      if (this.editor.instanceState.isCoarsePointer) {
        const snappedToRightAngle = snapAngle(newSelectionRotation, 4);
        const angleToRightAngle = angleDelta(newSelectionRotation, snappedToRightAngle);
        if (Math.abs(angleToRightAngle) < degreesToRadians(5)) {
          newSelectionRotation = snappedToRightAngle;
        }
      }
    }
    return newSelectionRotation - initialSelectionRotation;
  }
}
export {
  Rotating
};
//# sourceMappingURL=Rotating.mjs.map
