import {
  assert,
  exhaustiveSwitchError,
  useEditor,
  useValue
} from "@tldraw/editor";
function compactMenuItems(arr) {
  return arr.filter((i) => i !== void 0 && i !== null && i !== false);
}
function menuGroup(id, ...children) {
  const childItems = compactMenuItems(children);
  if (childItems.length === 0)
    return null;
  return {
    id,
    type: "group",
    checkbox: childItems.every((child) => child.type === "item" && child.actionItem.checkbox),
    disabled: childItems.every((child) => child.disabled),
    readonlyOk: childItems.some((child) => child.readonlyOk),
    children: childItems
  };
}
function menuSubmenu(id, label, ...children) {
  const childItems = compactMenuItems(children);
  if (childItems.length === 0)
    return null;
  return {
    id,
    type: "submenu",
    label,
    children: childItems,
    disabled: childItems.every((child) => child.disabled),
    readonlyOk: childItems.some((child) => child.readonlyOk)
  };
}
function menuCustom(id, opts = {}) {
  const { readonlyOk = true, disabled = false } = opts;
  return {
    id,
    type: "custom",
    disabled,
    readonlyOk
  };
}
function menuItem(actionItem, opts = {}) {
  if (!actionItem) {
    throw Error("No action item provided to menuItem");
  }
  if (!actionItem.label) {
    throw Error("Trying to create menu item for action item that doesn't have a label");
  }
  const { checked = false, disabled = false } = opts;
  return {
    id: actionItem.id,
    type: "item",
    actionItem,
    disabled,
    checked,
    readonlyOk: actionItem.readonlyOk
  };
}
function shapesWithUnboundArrows(editor) {
  const { selectedShapeIds } = editor;
  const selectedShapes = selectedShapeIds.map((id) => {
    return editor.getShape(id);
  });
  return selectedShapes.filter((shape) => {
    if (!shape)
      return false;
    if (editor.isShapeOfType(shape, "arrow") && shape.props.start.type === "binding") {
      return false;
    }
    if (editor.isShapeOfType(shape, "arrow") && shape.props.end.type === "binding") {
      return false;
    }
    return true;
  });
}
const useThreeStackableItems = () => {
  const editor = useEditor();
  return useValue("threeStackableItems", () => shapesWithUnboundArrows(editor).length > 2, [editor]);
};
const useAllowGroup = () => {
  const editor = useEditor();
  return useValue("allowGroup", () => shapesWithUnboundArrows(editor).length > 1, [editor]);
};
const useAllowUngroup = () => {
  const editor = useEditor();
  return useValue(
    "allowUngroup",
    () => editor.selectedShapeIds.some((id) => editor.getShape(id)?.type === "group"),
    []
  );
};
function findMenuItem(menu, path) {
  const item = _findMenuItem(menu, path);
  assert(item, `Menu item ${path.join(" > ")} not found`);
  return item;
}
function _findMenuItem(menu, path) {
  const [next, ...rest] = path;
  if (!next)
    return null;
  const item = menu.find((item2) => item2.id === next);
  if (!item)
    return null;
  switch (item.type) {
    case "group":
    case "submenu":
      return rest.length === 0 ? item : _findMenuItem(item.children, rest);
    case "item":
    case "custom":
      return rest.length === 0 ? item : null;
    default:
      exhaustiveSwitchError(item, "type");
  }
}
const showMenuPaste = typeof window !== "undefined" && "navigator" in window && Boolean(navigator.clipboard) && Boolean(navigator.clipboard.read);
export {
  compactMenuItems,
  findMenuItem,
  menuCustom,
  menuGroup,
  menuItem,
  menuSubmenu,
  showMenuPaste,
  useAllowGroup,
  useAllowUngroup,
  useThreeStackableItems
};
//# sourceMappingURL=menuHelpers.mjs.map
