# As much as possible, each code line is made up as a single command.
# Similar commands are grouped under common names, and included as modules.
source ./log.nu
source ./code.nu

def daily [ place: path, ...labels: string ] { cd $place
  let label = $place | path expand | path basename
  let day = bash -c "date -I"
  let name = [ $day ...($labels) md ] | str join "."
  let node = ([ $place $name ] | path join | path expand)
  touch $node; c ab $node; ch $node; cap #$node
  code c -m $"[daily] [($label)] ($day) - ($labels | str join ".")"
}

def gc [--dura (-d): string = 2d, --old (-o)] {
  let call = if ($old) { [ --delete-old ]
  } else { [ --delete-older-than $dura ] }
  log -l garbage.nix sudo nix-collect-garbage ...$call
}

def md [path: path] { nsh glow glow $path '-p' }

def hashes [...loci: path] {
  let error = (1..7) | each {"# error #"} | str join ''
  $loci | each { |$locus|
    tree -afix --noreport $locus
    | lines | where { ($in | path type) == 'file' }
    | each {|n| [
      ( try { open --raw $n | hash sha256 } catch { $error }
      ), $n
    ] | to json -r }
    | each { print }
  } | flatten | str join "\n"
}

def guide [command: string] {
  nu -e $"($command) --help | ansi strip | hx"
}

def renu [...names: string] { $names | each { |n|
  http get $"https://share.operand.online/gram/build/config/nushell/($n).nu" |
  save -f ($"~/.config/nushell/($n).nu" | path expand)
} }

def prune [...shapes: glob] {
  tree -dafix --noreport ...($shapes) | lines | reverse | each { rmdir $in }
}

def either [original:any, second: any] { ([ $original $second ]| compact).0 }

def local [ range:string = "10.0.0.*" ] {
  nsh nmap nmap $range | split row "\n\n" | each {|r|
    $r | lines | get 0 | split column ' for ' | try { get column2 | get 0 }
} }

def scope --wrapped [ node: path = .,
  --code (-c),
  ...command: string,
] { cd $node; if ( $command | is-empty ) { change . } else { (...$command) }
  if $code { cap; try { code c } } }

# `change` happens so much, you can be quicker by using `ch`.
def change --wrapped [ ...nodes: path ] { hx ...($nodes | path expand) }
def ch --wrapped [ ...nodes: path ] { hx ...($nodes | path expand) }

def loc [] { hyprlock }

def scene [] {
  firefox 'https://paper.garden/#/view/9c585a14-68e8-81eb-8003-75348e700227?page-id=88655af3-d07d-8058-8004-56a921c9f8ba&section=interactions&index=0&share-id=1079b0fe-d8fd-81eb-8004-56bfebec1806'
}

def shrug [] { echo '¯\_(ツ)_/¯' }
def again [] { nu -c (atuin history list --session --cmd-only | rg -v again | tail -n 1) }

def descend [
  -l: int = 0,
  -c: string = '.git',
  command: string
] { if ($l > 0) { ls -d | where type == dir | get name | par-each {|d| cd $d;
  if ( [ '.' $c ] | path join | path exists ) { nu -c $command
  } else { descend -l ($l - 1) -c $c $command } } } }

def "descend log" [
  -l: int = 0,
] { let node = $"./(clock | str trim).descend.log"
  let graphs = descend -l $l -c .git 'print ""; git remote -v; print "---"; git fetch -p; git log --graph --all --format=format:"%h %G? (%ai) %aN: %s | %d" --abbrev-commit --date=relative -n 120'
  $graphs | str join "\n\n---\n\n" | save -f $node; ch $node
}

# def remember [name:string] {
#   let cmd = atuin history list --session --cmd-only | rg -v again | tail -n 2 | head -n 1;
#   echo $"def ($name) [] { ($cmd) }" | save --append ~/.build/config/nushell/command.nu }

# HacDC Shelf animations
def sl [anim:string] { ssh hacdc@10.11.4.197 $'echo ($anim) | nc localhost 5000'; }
def ss [anim:string] { ssh hacdc@10.11.4.237 $'echo ($anim) | nc localhost 5000'; }
def s [ side:string, anim:string ] {
  if ($side == 'a') { ssh hacdc@10.11.4.237 $'echo ($anim) | nc localhost 5000';
  } else { ssh hacdc@10.11.4.197 $'echo ($anim) | nc localhost 5000'; } }
def zoom [delay:duration = 60sec] { loop {
  [ a b ] | par-each { |s| s $s lightsaber_off }
  [ a b ] | par-each { |s| s $s lightsaber_on }
  [ a b ] | par-each { |s| s $s let_me_in_anim }
  [ a b ] | par-each { |s| s $s flicker_off }
  [ a b ] | par-each { |s| s $s let_me_in_anim }
  [ a b ] | par-each { |s| s $s flicker_on }
  sleep $delay;
} }

# -pass all channels:
# hide() { for node in $@; do mv $node .$node; done; }
# open() { xdg-open $@; }

def slp [] { systemctl suspend; swaylock; }
def comm [] { [
  "element-desktop" "signal-desktop" "webcord" "whatsapp-for-linux" "zulip"
  ] | par-each { |cmd| (nu -c $cmd) } }
def conn [] {
  ip a | rg '^\d+: ' | lines |
  each {|l| $l | split column ':' | get 0 | get column2 } |
  each {|conn| sudo ethtool $conn }
  sudo systemctl restart systemd-networkd; }

def mod [] { (cat ~/.build/nixos/module.nix | lines) }

def pstree [] { nix-shell -p pstree --command 'pstree -U'
  | str replace -ar '/nix/store/(\w{6})\w+-(\w+)-(.+)/' "/nix/store/$1*-$2-*/"; }

# Gstreamer hacks
def gst-pkg [...grades:string] {
  "gstreamer" ++ ($grades | each {|grade| $"gst-plugins-($grade)"})
  | each {|pkg| $"gst_all_1.($pkg)" } }

def gin [commands: string] { (nix-shell -p ffmpeg ...(gst-pkg base good bad ugly)
  --command $"gst-inspect-1.0 ($commands)";) }

def chan [] { (nix-shell -p ...(gst-pkg base good)
  --command 'gst-launch-1.0 \
  audiotestsrc freq=220 ! audioconvert ! autoaudiosink \
  videotestsrc ! videoconvert ! autovideosink';) }

def noise [noise: string] { (nix-shell -p ...(gst-pkg base good bad)
  --command $'gst-launch-1.0 \
  audiotestsrc wave=($noise) ! audioconvert ! autoaudiosink \
  audiotestsrc wave=($noise) ! spacescope ! videoconvert ! autovideosink';) }

# manage processes
def pause   [command:string, --dura (-d): duration] {
  ps | where name =~ $command | get pid | str join ' ' | xargs kill -STOP;
  if not ($dura | is-empty) { sleep $dura; unpause $command; }
}
def unpause [command:string] { ps | where name =~ $command | get pid | str join ' ' | xargs kill -CONT; }
def hold [program:string pause:duration] { pause $program; sleep $pause; unpause $program; }
def process [command:string] { ps | where name =~ $command; }

# log() { echo "cd $PWD && $@ # $(date -Is)" >> ~/.build/log;
#   echo "$($@)" | ruby -e 'puts STDIN.readlines.map{|l|"# #{l}"}.join' |
#     tee -a ~/.build/log; echo >> ~/.build/log; }

# -1 channel: node place:
# bin() { change ~/bin/$1; chmod +x ~/bin/$1; }
# cmd() { ~/base/cmd/$1/cmd/$2; }
# co() { cd ~/company/$1; }
# domain() { cd ~/place/$1/domain/; }
# hex() { cd ~/base/hex/$1; }
# place() { cd ~/place/$1; }
# replace() { mkdir -p $(dirname $1); change $1; }
# run() { mkdir $1; cd $1; }

# query
def shape [cmd: string] { which $cmd; }

# access() { cat log/*access* | rg 'GET / ' | rg -v -i googlebot | rg -v 'paloaltonetworks.com'; }
def address [] { http get "https://api.ipify.org" }
# addresses() { ssh root@mail.assembled.app -t 'cd /mailu && docker-compose exec admin flask mailu config-export user.email'; }
def b [] { fg; }
def mark [] { ssh root@10.11.99.1 ./goMarkableStream; }

# def rebuild [] { ch ~/.build/nixos/*.nix; c sudo nixos-rebuild switch; }
def reslice [] { sudo systemctl restart display-manager; } # user.slice
def channel [] { nix-shell -p lsof --command 'sudo lsof -nP -iTCP -sTCP:LISTEN' }
def issue [] { scope ~/issue }

# -quickies
# burn() { open -a Firefox; }
# chec() { change ~/chec.plain; }
# cloc() { nix-shell -p cloc --command 'cloc .'; }
# dns() { sudo resolvectl flush-caches; }
# gp() { gpaste-client > /dev/null; }
# kaboom() { c git k deploy main && git k origin main; }
# llm() { cd ~/place/llm/gpt4all/chat; ./gpt4all-lora-quantized-linux-x86; }
# mixup() { mix deps.compile; iex -S mix run; }
# pc() { podman-compose; }
# pclean() { podman ps -a | ag '(Created|Exited)' | ag -o '\w+$' | xargs podman rm; }
# phrase() { session run reap; cd ~/.cache.channel/etymonline/phrases/; }
# pkill() { podman ps -a | ag 'Up ' | ag -o '\w+$' | xargs podman kill; }
# pose() { g clone ~/place/assemble/pose; }
# prun() { podman run; }
# q() { clear; ls; echo; g s; echo; g lg -n 6; echo; gap; }
# relay() { python3 -m http.server; }
# remix() { mix deps.clean --all; mix deps.get; mix deps.update --all; mix deps.get; mix deps.compile; }
# x() { xplr; }
# zu() { zulip-term; }

def pond [] { mkdir ~/pond; nix-shell -p seaweedfs --command "weed mount -filer=10.10.0.11:8888 -dir ~/pond -allowOthers=false"; }
# def lake [] { mkdir ~/lake; nix-shell -p seaweedfs --command "weed mount -filer=10.10.0.11:8888 -dir ~/lake -allowOthers=false"; }

# # more
# bl() { history | tail -n "+$(\
#   history | nsh fzf fzf --tac | sed -r 's/^ ([0-9]+).*$/\1/'\
#   )" | sed -r 's/^ [0-9]+ +(.*)$/\1/'; }
# passed() {
#   if [ "$1" == "" ]; then
#   cache=".passed.$(clock).command.index"; history > $cache; change $cache; else
#   history | tail -n $1; fi; }

def policy [address: string] {
  let domain = (nix-shell -p ruby --command $"ruby -r uri -e \"puts URI.parse\('($address)'\).host\"")
  let node = (nix-shell -p ruby --command $"ruby -r uri -e \"puts URI.parse\('($address)'\).path.gsub\(/\\/$/, ''\)\"")
  let place = $"($domain)(dirname $node)";
  ( cd ~/base/code/policy; if not ($place | path exists) { mkdir $place };
    wget $address -O $"($domain)($node)";
    git add $place; git commit -m $"policy '($address)'"; git push;); }


def pull [address: string, link: string] {
  ssh $address "sudo chmod -R 755 /var/lib/transmission/Downloads && sudo chmod 755 /var/lib/transmission/"
  bind media media; rsync -av $"access@($address):/var/lib/transmission/Downloads/*" media/magnesis/; unbind media }

# ## Docker
# # drun() { sudo docker run; }
# # dc() { sudo docker-compose; }
# # dclean() { sudo docker ps -a | ag '(Created|Exited)' | ag -o '\w+$' | sudo xargs docker rm; }
# # dkill() { sudo docker ps -a | ag 'Up ' | ag -o '\w+$' | sudo xargs docker kill; }

# # prepare() {
# #   mkdir -p ./log
# #   rsync access@program:/var/log/nginx/$(basename $(pwd))-access.log \
# #     ./log/$(basename $(pwd)).$(date -Is | sed 's/:/-/g').access.log
# #   rsync access@program:/var/log/nginx/$(basename $(pwd))-error.log \
# #     ./log/$(basename $(pwd)).d$(date -Is | sed 's/:/-/g' | sed 's/T/c/g' ).error.log
# # }

# # logs() {
# #   clock=`date -Iseconds`
# #   node=~/place/program/log/$clock
# #   mkdir -p $node
# #   rsync access@10.0.0.129:/var/log/nginx/ $node/ &&\
# #     ssh access@10.0.0.129 sudo rm -rf /var/log/nginx/*
# #   cd $node
# # }

# # nodelines() { cat /proc/sys/fs/inotify/max_user_watches;
# #   [[ -z "$1" ]] && return
# #   echo fs.inotify.max_user_watches=$1 | sudo tee -a /etc/sysctl.conf && sudo sysctl -p; }

# muscle memory!
def bush [] { tree -af --gitignore -i -I .git }
def "qu medi" [] { nu -e $"(cat qu/medi.nu)" }

def keys [] {
  group-by { |n| md5sum | str replace "  -" "" } |
  table --expand
}

# error: misaligned because a blank column appears in:
# zellij list-sessions | columnar | rename session age bind
def columnar [--blank (-b)] {
  if ($blank) {
    lines | ansi strip | split column -r "(\t+|  +)"
  } else {
    lines | ansi strip |
    str replace -ra ' *\[(.+)\] *' '|$1' |
    str replace -ra ' *\<(.+)\> *' '|$1' |
    str replace -ra ' *\((.+)\) *' '|$1' |
    split column '|'
} }

def "slide md" [
  $node: path,
  --delay (-d): duration = 12sec
] {
  let recipe = cat $node | split row "\n```\n" | split column "\n```nu\n";
  loop { 0..(($recipe | length) - 1) | each {|n|
    clear; $recipe |
      get $n | transpose |  get column1 |
      str join "\n>>> nu\n\n" |  print;
      sleep $delay;
} } }

def send [
  source: list<string>,
  aim: path,
  --plan (-p),
] {
  print $"@-> ($aim)"; print $source; mkdir ($aim | path dirname)
  if (not $plan) { time rsync -av ...($source) ($aim) }
}
