#!/usr/bin/env nu
source ~/.nue/nix.nu
source ~/.nue/disc.nu
source ~/.nue/grammar.nu

let mem = '32GB'
let swap = [ ('-' + $mem) '100%' ]

let boot = [ '1MB' '512MB' ]
let nix = [ ($boot | last) '128GB' ]
let root = [ ($nix | last) '256GB' ]
let home = [ ($root | last) ($swap | first) ]

def part [disc: string, part: any = ''] {
  ($disc | into string) + ($part | into string) }

def --wrapped bind [label: string, loc: path, ...call] {
  if not ($loc | path exists) { sudo mkdir $loc };
  sudo mount ...($call) $label $loc
}

def "bind all" [disc: string] {
  bind (part $disc 3) /mnt
  bind (part $disc 1) /mnt/boot -o umask=077
  bind (part $disc 2) /mnt/nix
  bind (part $disc 4) /mnt/home
}

def "flash" [disc: string] {
  nsh parted sudo parted $disc -- mklabel gpt

  nsh parted sudo parted $disc -- mkpart ESP fat32 ...($boot)
  nsh parted sudo parted $disc -- set 1 esp on
  sudo mkfs.fat -F 32 -n BOOT (part $disc 1)

  nsh parted sudo parted $disc -- mkpart nix ext4 ...($nix)
  sudo mkfs.ext4 -L nix (part $disc 2)

  nsh parted sudo parted $disc -- mkpart root ext4 ...($root)
  sudo mkfs.ext4 -L nixos (part $disc 3)

  nsh parted sudo parted $disc -- mkpart home ext4 ...($home)
  sudo mkfs.ext4 -L home (part $disc 4)

  nsh parted sudo parted $disc -- mkpart swap linux-swap ...($swap)
  sudo mkswap -L swap (part $disc 5)
}

def unbind [] {
  try { sudo umount /mnt/boot }
  try { sudo umount /mnt/nix }
  try { sudo umount /mnt/home }
  try { sudo umount /mnt }
}

def "clone home" [--user (-u): string = nixos] {
  let home = if ($user == 'root') {
    '/mnt/root' } else {
    $"/mnt/home/($user)" }
  | path expand
  if not ($home | path exists) { sudo mkdir $home }
  $home
}

def "clone base" [base: string, --user (-u): string = nixos, --loc (-l): path] {
  let loc = $loc | assure { [(clone home -u $user) $base] | path join }
  # if ($loc | path exists) { print $"Oh no! ($loc) is already loaded."; return;
  # } else {
  try { nsh gitoxide sudo gix clone $"~/($base)" $loc }
  # }
}

def "clone loc" [base: path, --user (-u): string = nixos] {
  let loc = [(clone home -u $user) $base] | path join
  nsh rsync sudo rsync -av ([~ $base] | path join) (clone home -u $user)
}

def "clone all" [--user (-u): string = nixos] {
  clone home -u $user
  clone base .nue -u $user
  clone base .nux -u $user
  clone base /pool -u $user -l /mnt

  # clone base .nue -u $user
  # clone base .nux -u $user
  # clone base .mech -u $user
  # clone base page -u $user
  # clone base diagram -u $user

  # clone loc -u $user share
  # clone loc -u $user gram
}

def "label" [disc: string, part: any] {
  (ls /dev/disk/by-label).name | where {|id| ($id | path expand) == (part $disc $part) } | first
}

def "nix bind" [disc: string, part: int, bind: string, aim: string, --form (-f): string = ext4] {
  $"fileSystems.''($bind)'' = { device = ''/dev/($aim)($part # label $disc $part
  )''; fsType = ''($form)''; };" | str replace -a "''" '"'
}

def "nix swap" [disc: string, part: int] {
  $"swapDevices = [
    { device = ''(label $disc $part)''; options = [ ''discard=once'' ]; }
  ];" | str replace -a "''" '"'
}

def "nix machine" [disc: string, arch: string, kernel: string, aim: string] {
  ([ (nix preamble)
    (nix arch $arch $kernel)
    (nix bind $disc 1 "/boot" $aim -f vfat)
    (nix bind $disc 2 "/nix" $aim)
    (nix bind $disc 3 "/" $aim)
    (nix bind $disc 4 "/home" $aim)
    # (nix swap $disc 5)
  ] | flatten | each { "  " + $in } | str join "\n" | str trim) + "\n}\n"
}

def "nix preamble" [] {
'' + `{ config, lib, modulesPath, ... }:
{
  imports = [
    (modulesPath + "/installer/scan/not-detected.nix")
  ];

  boot.loader = { systemd-boot.enable = true; efi.canTouchEfiVariables = true; };
  boot.initrd.availableKernelModules = [ "xhci_pci" "thunderbolt" "nvme" "usb_storage" "usbhid" "sd_mod" ];
  boot.initrd.kernelModules = [ ];
  boot.extraModulePackages = [ ];
`  
}

def "nix arch" [arch: string, kernel: string] {
  [ $"nixpkgs.hostPlatform = lib.mkDefault \"($arch)-($kernel)\";" ] ++ (
  if ($arch == 'x86_64') { [
    "hardware.cpu.intel.updateMicrocode = lib.mkDefault config.hardware.enableRedistributableFirmware;"
    "boot.kernelModules = [ ''kvm-intel'' ];"
  ] } else {
    "boot.kernelModules = [];" })
}

def main [disc?: string, --recycle (-r) ] { timeit { try {
  unbind
  let disc = $disc | assure { "/dev/" + (disc choose) }

  if not $recycle { flash $disc }
  nsh parted sudo parted $disc -- p

  bind all $disc

  print "choose user:"
  let user = [ nixos root (whoami) '....' ] | choose
  let user = $user | replace ($user == '....') { input "user name:\t" }

  disc ls | where source =~ $disc | print
  tree -afFix /mnt | print
  "Copying in codebases." | print
  clone all -u $user

  let arch = [ x86_64 aarch64 ] | choose
  print ("Choose arch:    " + $arch)

  # darwin
  let kernel = [ linux ] | choose
  print ("Choose kernel:  " + $kernel)

  let machine = [ baseboard chesapeake pebble clam ] | choose
  print ("Choose machine: " + $machine)

  let aim = [ nvme0n1p nvme0n2p sda sdb sdc '....' ] | choose
  let aim = $aim | replace ($aim == '....') { input "disc name:\t" }
  print ("disc aim:      " + $aim)

  sudo nu -c $"(nix machine $disc $arch $kernel $aim | to nuon) | save -f '(clone home -u $user)/.nux/cell/($machine)/machine.nix'"
  # nix machine $disc $arch $kernel | sudo do { save -f $"(clone home -u $user)/.nux/cell/($machine)/machine.nix" }

  # (sudo nixos-install
  #   --flake $"(clone home -u $user)/.nux#($machine)"
  #   -E 'with import <nixpkgs> { crossSystem = { config = "armv6l-unknown-linux-gnueabi"; }; };'
  # )
  cd $"(clone home -u $user)/.nux"
  sudo nix flake update
  sudo nixos-install --flake $"(clone home -u $user)/.nux#($machine)"
  }; unbind
} }
