{ lib, pkgs, ... }:
{
  # home.packages = with pkgs; [ helix-gpt ];

  # programs.git.extraConfig.core.editor = lib.mkOverride 100 "hx";
  programs.helix = {
    enable = true;
    settings = {
      editor.cursor-shape = {
        normal = "block";
        insert = "bar";
        select = "underline";
      };

      keys.normal = {
        L = "extend_to_line_end";
        H = "extend_to_line_start";
        space.w = ":w";
        space.q = ":q";
        space.x = ":x";
      };
    };
      # theme = "autumn_night"; #_transparent";
      # lsp.display-messages = true;
      # whitespace.render = "all";
      # whitespace.characters = { space = "·"; nbsp = "⍽"; tab = "→"; newline = "⤶"; };

    languages = with pkgs; {
      language-server = {
        # gpt = {
        #   command = "helix-gpt";
        #   args = [ "--handler" "ollama" "--ollamaModel" "codegemma:7b" ];
        # };

        deno = {
          command = "${pkgs.deno}/bin/deno";
          args = [ "lsp" ];
        };

        rust-analyzer = {
          config.rust-analyzer = {
            cargo.loadOutDirsFromCheck = true;
            checkOnSave.command = "clippy";
            procMacro.enable = true;
            lens = { references = true; methodReferences = true; };
            completion.autoimport.enable = true;
            experimental.procAttrMacros = true;
          };
        };
        eslint = {
          command = "vscode-eslint-language-server";
          args = [ "--stdio" ];
          config = {
            validate = "on";
            packageManager = "yarn";
            useESLintClass = false;
            codeActionOnSave.mode = "all";
            # codeActionsOnSave = { mode = "all"; };
            format = true;
            quiet = false;
            onIgnoredFiles = "off";
            rulesCustomizations = [ ];
            run = "onType";
            # nodePath configures the directory in which the eslint server should start its node_modules resolution.
            # This path is relative to the workspace folder (root dir) of the server instance.
            nodePath = "";
            # use the workspace folder location or the file location (if no workspace folder is open) as the working directory

            workingDirectory.mode = "auto";
            experimental = { };
            problems.shortenToSingleLine = false;
            codeAction = {
              disableRuleComment = {
                enable = true;
                location = "separateLine";
              };
              showDocumentation.enable = true;
            };
          };
        };
        nil = {
          command = "${pkgs.nil}/bin/nil";
          config.nil = {
            formatting.command = [ "${nixpkgs-fmt}/bin/nixpkgs-fmt" ];
            nix.flake.autoEvalInputs = true;
          };
        };
        typescript-language-server = {
          command = "${nodePackages.typescript-language-server}/bin/typescript-language-server";
          args = [ "--stdio" "--tsserver-path=${nodePackages.typescript}/lib/node_modules/typescript/lib" ];
          config.documentFormatting = false;
        };
        lexical = {
          command = "${pkgs.lexical}/bin/lexical";
          config.lexical = {
            # formatting.command = [ "${nixpkgs-fmt}/bin/nixpkgs-fmt" ];
          };
        };
      };
      language = let
        jsTsWebLanguageServers = [ "eslint" # "gpt"
          # { name = "deno"; }
          { name = "typescript-language-server"; except-features = [ "format" ]; }
          { name = "efm-lsp-prettier"; only-features = [ "format" ]; }
        ];
        indent = { unit = "  "; tab-width = 2; };
      in [

        { name = "bash"; file-types = [ "sh" "bash" ]; auto-format = true;
          # Indent with 2 spaces, simplify the code, indent switch cases, add space after redirection
          formatter = { command = "${pkgs.shfmt}/bin/shfmt"; args = [ "-i" "2" "-s" "-ci" "-sr" ]; };
          # language-servers = ["gpt"];
        }

        { name = "typescript"; language-servers = jsTsWebLanguageServers; inherit indent; }
        { name = "javascript"; language-servers = jsTsWebLanguageServers; inherit indent; }
        { name = "jsx"; language-servers = jsTsWebLanguageServers; inherit indent; }
        { name = "tsx"; language-servers = jsTsWebLanguageServers; inherit indent; }

        # { name = "vue"; language-servers = [{ name = "vuels"; except-features = [ "format" ]; } { name = "efm-lsp-prettier"; } "eslint"]; inherit indent; }

        { name = "elixir"; file-types = ["ex" "exs" ]; formatter.command = "${pkgs.elixir}/bin/mix format"; inherit indent; language-servers = [ "lexical" ]; } # "gpt" ]; }
        { name = "heex"; language-servers = [ "lexical" ]; inherit indent; }
        { name = "rust"; auto-format = false; file-types = [ "lalrpop" "rs" ]; inherit indent; language-servers = [ "rust-analyzer" ]; } # "gpt" ]; }
        # { name = "ruby"; file-types = [ "rb" "rake" "rakefile" "irb" "gemfile" "gemspec" "Rakefile" "Gemfile" "Fastfile" "Matchfile" "Pluginfile" "Appfile" ]; inherit indent; }
        # { name = "zig"; auto-format = false; language-servers = [ "gpt" ]; inherit indent; }
        # { name = "nim"; auto-format = false; language-servers = [ "gpt" ]; inherit indent; }

        { name = "sql"; formatter.command = "pg_format"; inherit indent; }
        { name = "nix"; language-servers = [ "nil" ]; inherit indent; }
        { name = "json"; language-servers = [ { name = "vscode-json-language-server"; except-features = [ "format" ]; } "efm-lsp-prettier"]; inherit indent; } # "gpt"

        # human-readable
        { name = "markdown"; soft-wrap.enable = true;
          language-servers = [
            { name = "marksman"; except-features = [ "format" ]; }
            "ltex-ls" "efm-lsp-prettier"
          ];
        inherit indent; }

        { name = "xml"; file-types = [ "xml" "html" ];
          # auto-format = true;
          formatter = {
            command = "${pkgs.yq-go}/bin/yq";
            args = [ "--input-format" "xml" "--output-format" "xml" "--indent" "2" ];
          };
        inherit indent; }

      ];
    };

    # themes = {
      # autumn_night_transparent = { "inherits" = "autumn_night"; "ui.background" = {}; };
    # };

    # package = pkgs.helix.overrideAttrs (self: {
    #   makeWrapperArgs = with pkgs; self.makeWrapperArgs or [] ++ [
    #     "--suffix" "PATH" ":" (lib.makeBinPath [
    #       elixir-ls
    #       marksman
    #       nodePackages.yaml-language-server # yaml/json
    #       nodePackages.stylelint
    #       pgformatter
    #       taplo # toml
    #     ])
    #   ];
    # });
  };
}
