defmodule Op.BaseTest do
  use Op.DataCase

  alias Op.Base

  describe "queries" do
    alias Op.Base.Queri
    import Op.BaseFixtures

    @invalid_attrs %{code: nil, label: nil, signed: nil, cache: nil, source: nil, lang: nil, applies: nil, expires: nil, chronology: nil}

    test "list_queries/0 returns all queries" do
      queri = queri_fixture()
      assert Base.list_queries() == [queri]
    end

    test "get_queri!/1 returns the queri with given id" do
      queri = queri_fixture()
      assert Base.get_queri!(queri.id) == queri
    end

    test "create_queri/1 with valid data creates a queri" do
      valid_attrs = %{code: "some code", label: "some label", signed: "some signed", cache: "some cache", source: "some source", lang: "some lang", applies: ~U[2024-02-05 17:54:00Z], expires: ~U[2024-02-05 17:54:00Z], chronology: "some chronology"}

      assert {:ok, %Queri{} = queri} = Base.create_queri(valid_attrs)
      assert queri.code == "some code"
      assert queri.label == "some label"
      assert queri.signed == "some signed"
      assert queri.cache == "some cache"
      assert queri.source == "some source"
      assert queri.lang == "some lang"
      assert queri.applies == ~U[2024-02-05 17:54:00Z]
      assert queri.expires == ~U[2024-02-05 17:54:00Z]
      assert queri.chronology == "some chronology"
    end

    test "create_queri/1 with invalid data returns error changeset" do
      assert {:error, %Ecto.Changeset{}} = Base.create_queri(@invalid_attrs)
    end

    test "update_queri/2 with valid data updates the queri" do
      queri = queri_fixture()
      update_attrs = %{code: "some updated code", label: "some updated label", signed: "some updated signed", cache: "some updated cache", source: "some updated source", lang: "some updated lang", applies: ~U[2024-02-06 17:54:00Z], expires: ~U[2024-02-06 17:54:00Z], chronology: "some updated chronology"}

      assert {:ok, %Queri{} = queri} = Base.update_queri(queri, update_attrs)
      assert queri.code == "some updated code"
      assert queri.label == "some updated label"
      assert queri.signed == "some updated signed"
      assert queri.cache == "some updated cache"
      assert queri.source == "some updated source"
      assert queri.lang == "some updated lang"
      assert queri.applies == ~U[2024-02-06 17:54:00Z]
      assert queri.expires == ~U[2024-02-06 17:54:00Z]
      assert queri.chronology == "some updated chronology"
    end

    test "update_queri/2 with invalid data returns error changeset" do
      queri = queri_fixture()
      assert {:error, %Ecto.Changeset{}} = Base.update_queri(queri, @invalid_attrs)
      assert queri == Base.get_queri!(queri.id)
    end

    test "delete_queri/1 deletes the queri" do
      queri = queri_fixture()
      assert {:ok, %Queri{}} = Base.delete_queri(queri)
      assert_raise Ecto.NoResultsError, fn -> Base.get_queri!(queri.id) end
    end

    test "change_queri/1 returns a queri changeset" do
      queri = queri_fixture()
      assert %Ecto.Changeset{} = Base.change_queri(queri)
    end
  end
end
